define(["jquery", "Threedadv_Customer/js/fabric", "flatpickr"], function (
    $,
    fabric,
    flatpickr
) {
    return function (config) {
        const {
            seatTypes,
            zones,
            tablesInfo,
            imagesTablesUrls,
            tablesInfoId,
            imagesFloorAreaUrls,
        } = config;
        let counter = 0;
        let currentId;
        const FONT_SIZE_AREA = 15;
        const FONT_SIZE_FLOOR = 15;
        const RECTANGLE_SIZE_CONTROLS = 50;
        const ZONE_TYPE = {
            area: "AREA",
            floor: "Piso",
            table: "TABLE",
        };
        const FONT_SIZE_TABLES = 15;
        const ZONES_AVAILABLE = zones;

        const SEAT_TYPES = seatTypes;

        const NUMBER_OF_SEATS = [2, 4, 6, 8, 10, 12, 14];

        const ZONES_CONFIGURATION = [
            {
                name: "Barra",
                color: "#5568CD",
            },
            {
                name: "Planta Alta",
                color: "#AF5FC1",
            },
            {
                name: "Salón",
                color: "#C46552",
            },
            {
                name: "Bar",
                color: "#7EBFD2",
            },
            {
                name: "Terraza",
                color: "#7DD292",
            },
            {
                name: "Otro",
                color: "#E6B150",
            },
        ];

        const excludedNames = [
            "Baños",
            "Puerta grande",
            "Puerta chica",
            "Muro",
        ];

        const STATUS_TYPES = {
            ACTIVE: "active",
            INACTIVE: "inactive",
        };

        const getDefaultZone = () => {
            return ZONES_CONFIGURATION.find(
                (zone) => zone.name === ZONES_AVAILABLE[0]
            );
        };

        const TABLES_OPTIONS = [
            {
                id: "high-table",
                color: "rgba(0, 128, 0, 0.6)",
                name: "Mesa alta",
                src: imagesTablesUrls.tall,
                src_white: imagesTablesUrls.tall_white,
                src_black: imagesTablesUrls.tall_black,
            },
            {
                id: "square-table",
                color: "rgba(128, 128, 0, 0.6)",
                name: "Mesa cuadrada",
                src: imagesTablesUrls.square,
                src_white: imagesTablesUrls.square_white,
                src_black: imagesTablesUrls.square_black,
            },
            {
                id: "rectangular-table",
                color: "rgba(0, 128, 128, 0.6)",
                name: "Mesa rectangular",
                src: imagesTablesUrls.rectangular,
                src_white: imagesTablesUrls.rectangular_white,
                src_black: imagesTablesUrls.rectangular_black,
            },
            {
                id: "round-table",
                color: "rgba(0, 128, 0, 0.6)",
                name: "Mesa redonda",
                src: imagesTablesUrls.round,
                src_white: imagesTablesUrls.round_white,
                src_black: imagesTablesUrls.round_black,
            },
            {
                id: "bar",
                color: "rgba(0, 128, 128, 0.6)",
                name: "Barra",
                src: imagesTablesUrls.bar,
                src_white: imagesTablesUrls.bar_white,
                src_black: imagesTablesUrls.bar_black,
            },
            {
                id: "small-door",
                color: "rgba(0, 128, 128, 0.6)",
                name: "Puerta chica",
                src: imagesTablesUrls.small_door,
                src_white: imagesTablesUrls.small_door_white,
                src_black: imagesTablesUrls.small_door_black,
            },
            {
                id: "big-door",
                color: "rgba(0, 128, 128, 0.6)",
                name: "Puerta grande",
                src: imagesTablesUrls.big_door,
                src_white: imagesTablesUrls.big_door_white,
                src_black: imagesTablesUrls.big_door_black,
            },
            {
                id: "wall",
                color: "rgba(0, 128, 128, 0.6)",
                name: "Muro",
                src: imagesTablesUrls.wall,
                src_white: imagesTablesUrls.wall_white,
                src_black: imagesTablesUrls.wall_black,
            },
            {
                id: "wc",
                color: "rgba(0, 128, 128, 0.6)",
                name: "Baños",
                src: imagesTablesUrls.wc,
                src_white: imagesTablesUrls.wc_white,
                src_black: imagesTablesUrls.wc_black,
            },
        ];

        const loadJSON = function () {
            const map = tablesInfo;

            canvas.loadFromJSON(map, function () {
                const objects = canvas.getObjects();
                const textIndexes = {};
                objects.forEach((object, index) => {
                    if (object.type === "text") {
                        textIndexes[object.relationID] = index;
                        object.on("mouseup", selectTableForText);
                    }
                });
                objects.forEach((object, index) => {
                    if (object.type !== "text") {
                        object.set(
                            "textObject",
                            objects[textIndexes[object.relationID]]
                        );
                    }
                });
                canvas.renderAll();
            });
        };

        const updateZone = (event) => {
            const dataZone = ZONES_CONFIGURATION.find(
                (zone) => zone.name === event.target.value
            );
            if (!dataZone) return;
            const activeObj = canvas.getActiveObject();
            activeObj.set("fill", dataZone.color);
            const textObject = activeObj.textObject;
            if (textObject) {
                textObject.set("text", dataZone.name);
                const temp = createText({
                    label: dataZone.name,
                    fontSize: textObject.fontSize,
                });
                updatePositionText(activeObj);
            }
            canvas.renderAll();
        };
        const createOptionsZones = () => {
            const select = document.getElementById("typeZone");
            ZONES_AVAILABLE.forEach((zone) => {
                const opt = document.createElement("option");
                opt.value = zone;
                opt.innerHTML = zone;
                select.appendChild(opt);
            });
            select.addEventListener("change", updateZone);
        };

        createOptionsZones();

        const updateGroup = () => {
            const activeObj = canvas.getActiveObject();
            const textObject = activeObj.textObject;
            if (!textObject) return;

            const zoneType = activeObj.zoneType;

            if (zoneType === ZONE_TYPE.table) {
                textObject.set(
                    "text",
                    document.getElementById("tableText").value
                );

                const statusSelect = document.getElementById("statusSelect");
                if (statusSelect) {
                    const newStatus = statusSelect.value;
                    activeObj.set("status", newStatus);

                    if (newStatus === STATUS_TYPES.INACTIVE) {
                        if (window.flatpickrInstance) {
                            const selectedDates =
                                window.flatpickrInstance.selectedDates;
                            if (selectedDates.length === 2) {
                                activeObj.set(
                                    "startDate",
                                    formatDateTime(selectedDates[0])
                                );
                                activeObj.set(
                                    "endDate",
                                    formatDateTime(selectedDates[1])
                                );
                            }
                        }
                    } else {
                        activeObj.set("startDate", null);
                        activeObj.set("endDate", null);
                    }
                }
            } else if (zoneType === ZONE_TYPE.area) {
                textObject.set(
                    "text",
                    document.getElementById("typeZone").value
                );
            } else if (zoneType === ZONE_TYPE.floor) {
                textObject.set(
                    "text",
                    document.getElementById("floorText").value
                );
            }

            updatePositionText(activeObj);
            canvas.renderAll();
            unselectAction();
        };

        const saveJSON = () => {
            try {
                canvas.includeDefaultValues = false;
                const jsonData = canvas.toJSON([
                    "zoneType",
                    "relationID",
                    "selectable",
                    "numberOfSeats",
                    "seatType",
                    "status",
                    "startDate",
                    "endDate",
                    "element_name",
                ]);

                if (
                    !confirm(
                        "Al actualizar el mapa puede ocasionar que se pierdan los datos no sincronizados que tenga en la APP"
                    )
                ) {
                    return;
                }

                jsonData.objects.forEach((obj) => {
                    if (obj.startDate instanceof Date) {
                        obj.startDate = obj.startDate
                            .toISOString()
                            .split("T")[0];
                    }
                    if (obj.endDate instanceof Date) {
                        obj.startDate = formatDateTime(obj.startDate);
                    }
                });

                const stringCanvas = JSON.stringify(jsonData);

                const formId = tablesInfoId
                    ? "edit-tables-form"
                    : "create-tables-form";
                const inputId = tablesInfoId
                    ? "table_data_webview_edit"
                    : "table_data_webview_create";

                const form = document.getElementById(formId);
                const input = document.getElementById(inputId);

                if (!form || !input) {
                    throw new Error("Elementos del formulario no encontrados");
                }

                input.value = stringCanvas;
                form.submit();
            } catch (error) {
                console.error("Error al generar JSON:", error);
                alert(
                    "Ocurrió un error al preparar los datos para guardar: " +
                        error.message
                );
            }
        };

        const selectTableForText = (opt) => {
            const objects = canvas.getObjects();
            const object = objects.find((item) => {
                return item.relationID === opt.target.relationID;
            });
            if (object) {
                canvas.setActiveObject(object);
                canvas.renderAll();
            }
        };

        async function loadAndChangeSVGColor({
            id,
            color,
            name,
            src,
            src_white,
        }) {
            const svgGray = await fetchSVG(src);
            const svgWhite = await fetchSVG(src_white);

            const tempContainer = document.createElement("div");
            tempContainer.innerHTML = svgGray;
            tempContainer.innerHTML += svgWhite;

            tempContainer.addEventListener("dragstart", dragElement);

            addDraggableEvent(tempContainer);

            tempContainer.id = id;
            tempContainer.draggable = true;
            tempContainer.classList.add("table-control");

            const label = document.createElement("label");
            label.setAttribute("for", id);
            label.innerHTML = name;
            tempContainer.appendChild(label);
            document.getElementById("controls").appendChild(tempContainer);
        }

        function dragElement(e) {
            currentId = e.target.id;
            e.dataTransfer.setData("id", e.target.id);
        }

        const createControls = () => {
            TABLES_OPTIONS.forEach((table) => {
                loadAndChangeSVGColor(table);
            });

            const blocks = document.getElementsByClassName("floor-control");
            Array.from(blocks).forEach((rec) => {
                rec.addEventListener("dragstart", dragElement);
            });
        };

        createControls();
        fabric.Object.prototype.transparentCorners = false;
        const canvas = new fabric.Canvas("c1", {
            allowTouchScrolling: true,
        });

        function observe(eventName) {
            canvas.on(eventName, function (opt) {
                if (eventName === "drop") {
                    dropElement(opt.e);
                }
                if (
                    eventName === "object:scaling" ||
                    eventName === "object:moving" ||
                    eventName === "object:rotating"
                ) {
                    if (opt.target.type !== "activeSelection")
                        updatePositionText(opt.target);
                }
                if (
                    eventName === "selection:updated" ||
                    eventName === "selection:created"
                ) {
                    console.log("selection:updated,created");
                    canvas.allowTouchScrolling = false;
                    showSelection(opt);
                }
                if (eventName === "selection:cleared") {
                    console.log("selection:cleared");
                    canvas.allowTouchScrolling = true;
                    showOptionsSelection({ show: false });
                    document.querySelector("#info").innerHTML = "";
                }
            });
        }

        observe("drop");
        observe("selection:created");
        observe("selection:updated");
        observe("object:scaling");
        observe("object:moving");
        observe("object:rotating");
        observe("selection:cleared");
        observe("selected");
        observe("deselected");

        const optionsMap = {
            zone: "zoneSelectedOptions",
            table: "tableSelectedOptions",
            floor: "floorSelectedOptions",
        };

        const isVisibleId = (activeObj) => {
            const tableId = document.getElementById("table-id");

            if (
                activeObj.element_name &&
                excludedNames.includes(activeObj.element_name)
            ) {
                tableId.style.display = "none";
            } else {
                tableId.style.display = "block";
            }
        };

        const showOptions = (type) => {
            showOptionsSelection({ show: true });

            Object.values(optionsMap).forEach((id) => {
                document.getElementById(id).style.display = "none";
            });

            const selectedOption = optionsMap[type];
            if (selectedOption) {
                document.getElementById(selectedOption).style.display = "flex";
            }
        };

        const showOptionsSelection = ({ show }) => {
            const editSelection = document.getElementById("editSelection");
            const selectElement = document.getElementById("selectElement");
            editSelection.style.display = show ? "flex" : "none";
            selectElement.style.display = show ? "none" : "flex";
        };

        const setInputValue = (id, value) => {
            const input = document.getElementById(id);
            if (input) input.value = value;
        };

        const handleZoneTypeFloor = (activeObj, textObject) => {
            activeObj.lockRotation = true;
            activeObj.sendToBack();
            showOptions("floor");
            setInputValue("floorText", textObject.text);
        };

        const handleZoneTypeArea = (activeObj, textObject) => {
            activeObj.lockRotation = true;
            showOptions("zone");
            setInputValue("typeZone", textObject.text);
        };

        const handleZoneTypeTable = (activeObj, textObject) => {
            activeObj.bringToFront();
            activeObj.textObject.bringToFront();
            showOptions("table");
            setInputValue("numberOfSeats", activeObj.numberOfSeats);
            setInputValue("seatType", activeObj.seatType);
            setInputValue("tableText", textObject.text);

            if (activeObj.status) {
                setInputValue("statusSelect", activeObj.status);
            }

            isVisibleId(activeObj);
        };

        const updateSelectedInfo = async (activeObj) => {
            const selectedInfo = document.querySelector(".selected-info");
            if (activeObj && "element_name" in activeObj) {
                if (activeObj.element_name == "Piso") {
                    src = imagesFloorAreaUrls.floor_white;
                } else if (activeObj.element_name == "Area") {
                    src = imagesFloorAreaUrls.area_white;
                } else {
                    const table = TABLES_OPTIONS.find(
                        (table) => table.name === activeObj.element_name
                    );
                    src = table.src_white;
                }

                if (!src) return;

                const svg = await fetchSVG(src);
                const selectedTitle = document.querySelector("#selected-title");
                selectedTitle.textContent = activeObj.element_name;

                selectedInfo
                    .querySelectorAll("svg")
                    .forEach((svgElement) => svgElement.remove());

                const tempDiv = document.createElement("div");
                tempDiv.innerHTML = svg;
                const newSVG = tempDiv.firstElementChild;

                if (newSVG && newSVG.nodeName.toLowerCase() === "svg") {
                    selectedInfo.insertBefore(newSVG, selectedInfo.firstChild);
                }

                selectedInfo.style.display = "flex";
            } else {
                selectedInfo.style.display = "none";
            }
        };

        const showSelection = async (opt) => {
            const activeObj = canvas.getActiveObject();
            const textObject = activeObj?.textObject;

            if (!textObject) return;

            const zoneType = activeObj.zoneType;

            const statusSection = document.getElementById(
                "statusSelectedOptions"
            );
            statusSection.style.display =
                zoneType === ZONE_TYPE.table ? "flex" : "none";

            if (zoneType === ZONE_TYPE.table) {
                if (!window.flatpickrInstance) {
                    window.flatpickrInstance = initFlatpickr();
                }

                const currentStatus = activeObj.status || STATUS_TYPES.ACTIVE;
                document.getElementById("statusSelect").value = currentStatus;

                const dateRangeContainer =
                    document.getElementById("dateRangeContainer");
                if (currentStatus === STATUS_TYPES.INACTIVE) {
                    dateRangeContainer.style.display = "block";

                    if (activeObj.startDate && activeObj.endDate) {
                        try {
                            const startDate = parseDateWithoutTimezone(
                                activeObj.startDate
                            );
                            const endDate = parseDateWithoutTimezone(
                                activeObj.endDate
                            );

                            if (
                                startDate &&
                                endDate &&
                                !isNaN(startDate.getTime()) &&
                                !isNaN(endDate.getTime())
                            ) {
                                window.flatpickrInstance.setDate(
                                    [startDate, endDate],
                                    false
                                );
                            }
                        } catch (e) {
                            console.error("Error al procesar fechas:", e);
                        }
                    } else {
                        window.flatpickrInstance.clear();
                    }

                    activeObj.set({
                        opacity: 0.7,
                        shadow: "rgba(0,0,0,0.3) 2px 2px 10px",
                    });
                } else {
                    dateRangeContainer.style.display = "none";
                    activeObj.set({
                        opacity: 1,
                        shadow: "",
                    });
                }

                canvas.renderAll();
            }

            switch (zoneType) {
                case ZONE_TYPE.floor:
                    handleZoneTypeFloor(activeObj, textObject);
                    document.querySelector(
                        "#info"
                    ).innerHTML = `${textObject.text}`;
                    break;
                case ZONE_TYPE.area:
                    handleZoneTypeArea(activeObj, textObject);
                    document.querySelector(
                        "#info"
                    ).innerHTML = `${activeObj?.element_name}`;
                    break;
                case ZONE_TYPE.table:
                    handleZoneTypeTable(activeObj, textObject);
                    let infoText = `${activeObj?.element_name ?? ""} ${
                        textObject.text
                    }`;
                    document.querySelector("#info").innerHTML = infoText;
                    break;
                default:
                    return;
            }

            document.querySelector("#selection").style.display = "flex";
            await updateSelectedInfo(activeObj);
        };

        async function dropElement(e) {
            e.preventDefault();
            const id = currentId ?? e.dataTransfer.getData("id");
            let newObject;
            let label = "";
            let zoneType = "";
            let fontSize = "";
            const defaultZone = getDefaultZone();
            if (id === "area") {
                newObject = createRectangle({
                    x: e.layerX,
                    y: e.layerY,
                    color: "#DEDAD1",
                    rx: 5,
                    ry: 5,
                });
                zoneType = ZONE_TYPE.area;
                label = "Área";
                fontSize = FONT_SIZE_AREA;
                newObject.set("element_name", "Area");
            } else if (id === "floor") {
                newObject = createRectangle({
                    x: e.layerX,
                    y: e.layerY,
                    color: "#E3EDEB",
                    rx: 5,
                    ry: 5,
                });
                zoneType = ZONE_TYPE.floor;
                label = `${zoneType} ${counter.toString()}`;
                fontSize = FONT_SIZE_FLOOR;
                newObject.set("element_name", "Piso");
            } else {
                const data = TABLES_OPTIONS.find((table) => table.id === id);
                newObject = await loadSVGAsync({
                    svgURL: data.src_black,
                    x: e.layerX,
                    y: e.layerY,
                });
                zoneType = ZONE_TYPE.table;

                if (!excludedNames.includes(data.name)) {
                    label = counter.toString();
                }

                fontSize = FONT_SIZE_TABLES;
                newObject.set("numberOfSeats", NUMBER_OF_SEATS[0]);
                newObject.set("seatType", SEAT_TYPES[0]);
                newObject.set("element_name", data.name);
            }
            const relationID = Date.now();
            const text = createText({ label, fontSize });
            text.set("relationID", relationID);
            newObject.set("textObject", text);
            newObject.set("zoneType", zoneType);
            newObject.set("relationID", relationID);
            updatePositionText(newObject);
            counter++;
            canvas.add(newObject, text);
            canvas.renderAll();
        }

        function getRectBounds(rect) {
            const coords = rect.getCoords();
            const topMost = Math.min(
                coords[0].y,
                coords[1].y,
                coords[2].y,
                coords[3].y
            );
            const centerX = rect.getCenterPoint().x;
            return { topMost, centerX };
        }

        const updatePositionText = (object) => {
            const textObject = object.textObject;
            if (object.zoneType === ZONE_TYPE.table) {
                textObject.set({
                    left: object.getCenterPoint().x - textObject.width * 0.5,
                    top: object.getCenterPoint().y - textObject.height * 0.5,
                });
            } else {
                const { topMost, centerX } = getRectBounds(object);
                textObject.set({
                    left: centerX - textObject.width * 0.5,
                    top: topMost - textObject.fontSize,
                });
            }
        };

        function loadSVGAsync({ svgURL, x, y }) {
            return new Promise((resolve, reject) => {
                fabric.loadSVGFromURL(svgURL, (objects, options) => {
                    if (objects) {
                        const svg = fabric.util.groupSVGElements(
                            objects,
                            options
                        );

                        svg.set({
                            left: x - svg.width * 0.5,
                            top: y - svg.height * 0.5,
                        });

                        resolve(svg);
                    } else {
                        reject(new Error("Error al cargar el SVG"));
                    }
                });
            });
        }

        const createText = ({ label, fontSize }) => {
            const text = new fabric.Text(label, {
                fontSize,
                selectable: false,
                fontFamily: "Helvetica, Arial, sans-serif",
            });
            text.on("mouseup", selectTableForText);
            return text;
        };

        const createRectangle = ({ x, y, color, rx, ry }) => {
            return new fabric.Rect({
                width: RECTANGLE_SIZE_CONTROLS,
                height: RECTANGLE_SIZE_CONTROLS,
                left: x - RECTANGLE_SIZE_CONTROLS / 2,
                top: y - RECTANGLE_SIZE_CONTROLS / 2,
                fill: color,
                rx,
                ry,
            });
        };

        const removeSelected = () => {
            const activeObj = canvas.getActiveObject();
            const textObject = activeObj.textObject;
            canvas.discardActiveObject();
            if (textObject) {
                canvas.remove(textObject);
                canvas.remove(activeObj);
            }
            canvas.renderAll();
        };

        const updateSeatType = (event) => {
            const activeObj = canvas.getActiveObject();
            activeObj.set("seatType", event.target.value);
        };

        const createSeatTypeOptions = () => {
            const select = document.getElementById("seatType");
            SEAT_TYPES.forEach((type) => {
                const opt = document.createElement("option");
                opt.value = type;
                opt.innerHTML = type;
                select.appendChild(opt);
            });
            select.addEventListener("change", updateSeatType);
        };

        createSeatTypeOptions();

        const updateNumberOfSeats = (event) => {
            const activeObj = canvas.getActiveObject();
            activeObj.set("numberOfSeats", event.target.value);
        };

        const createNumberOfSeatsOptions = () => {
            const select = document.getElementById("numberOfSeats");
            NUMBER_OF_SEATS.forEach((type) => {
                const opt = document.createElement("option");
                opt.value = type;
                opt.innerHTML = type;
                select.appendChild(opt);
            });
            select.addEventListener("change", updateNumberOfSeats);
        };

        createNumberOfSeatsOptions();

        if (tablesInfo) {
            loadJSON();
        }

        document
            .getElementById("saveJSONButton")
            .addEventListener("click", saveJSON);
        document
            .getElementById("removeSelectedButton")
            .addEventListener("click", removeSelected);
        document
            .getElementById("updateGroupButton")
            .addEventListener("click", updateGroup);

        //draggable transition
        function addDraggableEvent(element) {
            element.addEventListener("dragstart", () => {
                element.classList.add("dragging");
            });

            element.addEventListener("dragend", () => {
                element.classList.remove("dragging");
            });
        }

        const floors = document.querySelectorAll(".floor-control");
        floors.forEach((element) => {
            addDraggableEvent(element);
        });

        async function fetchSVG(svgUrl) {
            try {
                const response = await fetch(svgUrl);
                if (!response.ok) throw new Error("Error al cargar el SVG");

                return await response.text();
            } catch (error) {
                console.error("Error al cargar el SVG:", error);
            }
        }

        //grid
        const grid = 5;
        $(document).ready(function () {
            const container = document.querySelector(".canvas-container #c1");
            if (container) {
                container.style.backgroundSize = `${grid}px ${grid}px`;
            }
        });

        canvas.on("object:moving", function (options) {
            const target = options.target;
            target.set({
                left: Math.round(target.left / grid) * grid,
                top: Math.round(target.top / grid) * grid,
            });
        });

        const unselectAction = () => {
            canvas.discardActiveObject();
            canvas.requestRenderAll();
        };

        // Unselect button
        document
            .getElementById("unselect")
            .addEventListener("click", unselectAction);

        const updateStatus = (event) => {
            const activeObj = canvas.getActiveObject();
            const newStatus = event.target.value;
            activeObj.set("status", newStatus);

            const dateRangeContainer =
                document.getElementById("dateRangeContainer");
            if (newStatus === STATUS_TYPES.INACTIVE) {
                dateRangeContainer.style.display = "block";

                if (activeObj.startDate && activeObj.endDate) {
                    window.flatpickrInstance.setDate([
                        new Date(activeObj.startDate),
                        new Date(activeObj.endDate),
                    ]);
                }
            } else {
                dateRangeContainer.style.display = "none";
                activeObj.set("startDate", null);
                activeObj.set("endDate", null);
                window.flatpickrInstance.clear();
            }
        };

        document
            .getElementById("statusSelect")
            .addEventListener("change", updateStatus);

        function parseDateWithoutTimezone(dateString) {
            if (!dateString) return null;
            const [datePart, timePart] = dateString.split(" ");
            if (!datePart || !timePart) return null;

            const [year, month, day] = datePart.split("-");
            const [hour, minute, second] = timePart.split(":");

            return new Date(year, month - 1, day, hour, minute, second);
        }

        const handleDateChange = (selectedDates) => {
            const activeObj = canvas.getActiveObject();
            if (selectedDates.length === 2) {
                const startDate = selectedDates[0];
                const endDate = selectedDates[1];

                activeObj.set("startDate", formatDateTime(startDate));
                activeObj.set("endDate", formatDateTime(endDate));

                canvas.renderAll();
            }
        };

        const initFlatpickr = () => {
            flatpickr.localize({
                months: {
                    longhand: [
                        "Enero",
                        "Febrero",
                        "Marzo",
                        "Abril",
                        "Mayo",
                        "Junio",
                        "Julio",
                        "Agosto",
                        "Septiembre",
                        "Octubre",
                        "Noviembre",
                        "Diciembre",
                    ],
                    shorthand: [
                        "Ene",
                        "Feb",
                        "Mar",
                        "Abr",
                        "May",
                        "Jun",
                        "Jul",
                        "Ago",
                        "Sep",
                        "Oct",
                        "Nov",
                        "Dic",
                    ],
                },
                weekdays: {
                    longhand: [
                        "Domingo",
                        "Lunes",
                        "Martes",
                        "Miércoles",
                        "Jueves",
                        "Viernes",
                        "Sábado",
                    ],
                    shorthand: [
                        "Dom",
                        "Lun",
                        "Mar",
                        "Mié",
                        "Jue",
                        "Vie",
                        "Sáb",
                    ],
                },
                rangeSeparator: " / ",
                firstDayOfWeek: 1,
            });

            return flatpickr("#dateRangePicker", {
                mode: "range",
                dateFormat: "Z",
                altInput: true,
                enableTime: true,
                altFormat: "d M y H:i:s",
                locale: "es",
                minDate: new Date().setHours(12, 0, 0, 0),
                onChange: handleDateChange,
            });
        };
    };

    function formatDateTime(date) {
        const pad = (n) => String(n).padStart(2, "0");
        return (
            date.getFullYear() +
            "-" +
            pad(date.getMonth() + 1) +
            "-" +
            pad(date.getDate()) +
            " " +
            pad(date.getHours()) +
            ":" +
            pad(date.getMinutes()) +
            ":" +
            pad(date.getSeconds())
        );
    }
});
