<?php
/**
 * Webkul Software.
 *
 * @category  Webkul
 * @package   Webkul_AdvancedBookingSystem
 * @author    Webkul Software Private Limited
 * @copyright Webkul Software Private Limited (https://webkul.com)
 * @license   https://store.webkul.com/license.html
 */
namespace Webkul\AdvancedBookingSystem\Block\Adminhtml\Catalog\Product\Edit\Tab;

use Magento\Backend\Block\Template\Context;
use Magento\Backend\Block\Widget;
use Magento\Framework\Registry;
use Magento\Framework\App\Request\DataPersistorInterface;
use Magento\Catalog\Model\Locator\LocatorInterface;

class Hotelbooking extends Widget
{
    /**
     * Reference to product objects that is being edited
     *
     * @var \Magento\Catalog\Model\Product
     */
    protected $_product = null;

    /**
     * Accordion block id
     *
     * @var string
     */
    protected $_blockId = 'hotelBookingInfo';

    /**
     * Registry
     *
     * @var Registry
     */
    protected $coreRegistry = null;

    /**
     * @var \Magento\Catalog\Model\Product\Attribute\Repository
     */
    protected $productAttributeRepository;

    /**
     * @var Magento\Directory\Model\ResourceModel\Region\CollectionFactory
     */
    protected $regionFactory;

    /**
     * @var \Magento\Backend\Model\UrlInterface
     */
    protected $backendUrl;

    /**
     * @var DataPersistorInterface
     */
    protected $dataPersistor;
    
    /**
     * @var LocatorInterface
     */
    protected $locator;
    
    /**
     * @var \Webkul\AdvancedBookingSystem\Helper\Data
     */
    protected $bookingHelper;

    /**
     * @param Context                                                         $context
     * @param Registry                                                        $registry
     * @param \Magento\Catalog\Model\Product\Attribute\Repository             $productAttributeRepository
     * @param \Magento\Directory\Model\ResourceModel\Region\CollectionFactory $regionFactory
     * @param \Magento\Backend\Model\UrlInterface                             $backendUrl
     * @param DataPersistorInterface                                          $dataPersistor
     * @param LocatorInterface                                                $locator
     * @param \Webkul\AdvancedBookingSystem\Helper\Data                       $bookingHelper
     * @param array                                                           $data
     */
    public function __construct(
        Context $context,
        Registry $registry,
        \Magento\Catalog\Model\Product\Attribute\Repository $productAttributeRepository,
        \Magento\Directory\Model\ResourceModel\Region\CollectionFactory $regionFactory,
        \Magento\Backend\Model\UrlInterface $backendUrl,
        DataPersistorInterface $dataPersistor,
        LocatorInterface $locator,
        \Webkul\AdvancedBookingSystem\Helper\Data $bookingHelper,
        array $data = []
    ) {
        $this->coreRegistry = $registry;
        $this->productAttributeRepository = $productAttributeRepository;
        $this->regionFactory = $regionFactory;
        $this->backendUrl = $backendUrl;
        $this->dataPersistor = $dataPersistor;
        $this->locator = $locator;
        $this->bookingHelper = $bookingHelper;
        parent::__construct($context, $data);
    }

    /**
     * Retrieve product
     *
     * @return \Magento\Catalog\Model\Product
     */
    public function getProduct()
    {
        if (!$this->locator->getProduct()->getId() && $this->dataPersistor->get('catalog_product')) {
            return $this->resolvePersistentData();
        }
        return $this->coreRegistry->registry('current_product');
    }

    /**
     * Resolve data persistence
     *
     * @param array $data
     * @return array
     */
    private function resolvePersistentData()
    {
        $persistentData = (array)$this->dataPersistor->get('catalog_product');
        $data = [];
        if (!empty($persistentData['product'])) {
            $data = $persistentData['product'];
            if (!empty($data['slot_data'])) {
                unset($persistentData['product']['slot_data']);
                $this->dataPersistor->set('catalog_product', $persistentData);
            }
        }
        return $data;
    }

    /**
     * _prepareLayout
     *
     * @return $this
     */
    protected function _prepareLayout()
    {
        $this->setData('opened', true);
        return parent::_prepareLayout();
    }

    /**
     * GetHotelCountryOptions
     *
     * @return array
     */
    public function getHotelCountryOptions()
    {
        try {
            $attribute = $this->productAttributeRepository->get('hotel_country');
            return $options = $attribute->getOptions();
        } catch (\Exception $e) {
            return [];
        }
    }

    /**
     * GetHotelStateOptions
     *
     * @param string $countryCode
     * @return object|boolean
     */
    public function getHotelStateOptions($countryCode)
    {
        try {
            return $this->regionFactory->create()->addCountryFilter($countryCode);
        } catch (\Exception $e) {
            return false;
        }
    }

    /**
     * GetRegionUpdateUrl
     *
     * @return string
     */
    public function getRegionUpdateUrl()
    {
        return $this->backendUrl->getUrl("directory/json/countryRegion");
    }

    /**
     * Return helper object
     *
     * @return \Webkul\AdvancedBookingSystem\Helper\Data
     */
    public function getBookingHelper()
    {
        return $this->bookingHelper;
    }
}
