<?php
/**
 * Webkul Software.
 *
 * @category  Webkul
 * @package   Webkul_AdvancedBookingSystem
 * @author    Webkul Software Private Limited
 * @copyright Webkul Software Private Limited (https://webkul.com)
 * @license   https://store.webkul.com/license.html
 */
namespace Webkul\AdvancedBookingSystem\Controller\Cancellation;

use Magento\Framework\App\Action\Action;
use Magento\Framework\App\Action\Context;
use Magento\Framework\App\RequestInterface;
use Magento\Framework\View\Result\PageFactory;
use Magento\Sales\Api\OrderRepositoryInterface;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Exception\InputException;
use Magento\Sales\Helper\Guest as GuestHelper;
use Magento\Framework\Controller\ResultInterface;

/**
 * Booking cancellation page controller
 */
class GuestIndex extends Action
{
    /**
     * @var PageFactory
     */
    protected $_resultPageFactory;
    
    /**
     * @var \Webkul\AdvancedBookingSystem\Helper\Data
     */
    protected $_helper;
    
    /**
     * @var OrderRepositoryInterface
     */
    protected $orderRepository;
    
    /**
     * @var GuestHelper
     */
    protected $guestHelper;

    /**
     * Constructor
     *
     * @param Context                                   $context
     * @param PageFactory                               $resultPageFactory
     * @param \Webkul\AdvancedBookingSystem\Helper\Data $helper
     * @param OrderRepositoryInterface                  $orderRepository
     * @param GuestHelper                               $guestHelper
     */
    public function __construct(
        Context $context,
        PageFactory $resultPageFactory,
        \Webkul\AdvancedBookingSystem\Helper\Data $helper,
        OrderRepositoryInterface $orderRepository,
        GuestHelper $guestHelper
    ) {
        $this->_helper = $helper;
        $this->orderRepository = $orderRepository;
        $this->guestHelper = $guestHelper;
        parent::__construct($context);
    }

    /**
     * Execute
     *
     * @return \Magento\Framework\Controller\Result\Page
     */
    public function execute()
    {
        $result = $this->guestHelper->loadValidOrder($this->getRequest());
        if ($result instanceof ResultInterface) {
            return $result;
        }

        if (!empty($this->getRequest()->getParam('id'))) {
            try {
                $itemId = $this->getRequest()->getParam('id');
                $item = $this->_helper->getOrderItem($itemId);
                $status = $this->_helper->getCancellationStatus($item->getOrderId(), $item->getId());
                $status = $this->_initOrder($item->getOrderId());
            } catch (\Magento\Framework\Exception\NoSuchEntityException $e) {
                $this->messageManager->addError($e->getMessage());
                $status = false;
            } catch (\Exception $e) {
                $status = false;
            }

            if ($status && $item->getIsCancellationAvailable() && $item->canRefund()) {
                $resultPage = $this->_resultPageFactory->create();
                $orderId = $this->getRequest()->getParam('order_id');
                $order = $this->orderRepository->get($orderId);
                $title = __('Booking Cancellation (Order # '. $order->getIncrementId().')');
                $resultPage->getConfig()->getTitle()->set($title);
                return $resultPage;
            } else {
                $this->messageManager->addError(__("Cancellation is not allowed"));
            }
        } else {
            $this->messageManager->addError(__("Order Item not Found"));
        }

        $resultRedirect = $this->resultRedirectFactory->create();
        return $resultRedirect->setPath('sales/guest/form/');
    }

    /**
     * Initialize order in registry
     *
     * @param int $id
     * @return boolean
     */
    protected function _initOrder($id = null)
    {
        try {
            $order = $this->orderRepository->get($id);
        } catch (NoSuchEntityException $e) {
            $this->messageManager->addErrorMessage(__('This order no longer exists.'));
            return false;
        } catch (InputException $e) {
            $this->messageManager->addErrorMessage(__('This order no longer exists.'));
            return false;
        }

        return true;
    }
}
