<?php
/**
 * Webkul Software.
 *
 * @category  Webkul
 * @package   Webkul_AdvancedBookingSystem
 * @author    Webkul Software Private Limited
 * @copyright Webkul Software Private Limited (https://webkul.com)
 * @license   https://store.webkul.com/license.html
 */
namespace Webkul\AdvancedBookingSystem\Helper\Admin;

use Webkul\AdvancedBookingSystem\Model\Info;

class Data extends \Webkul\AdvancedBookingSystem\Helper\Data
{
    /**
     * Process Booking Save
     *
     * @param array $data
     * @param \Magento\Catalog\Model\Product $product
     * @param \Magento\Sales\Model\Order\Item $item
     * @param int $isThrowError
     * @return void
     */
    public function processBookingSave($data, $product, $item, $isThrowError = 1)
    {
        if (empty($data['booking_date']) && empty($data['booking_time'])) {
            return null;
        }

        $error = 0;
        $currentTime = $this->getCurrentTime();
        $errorMessage = __('Invalid booking dates.');
        $selectedBookingDate = $data['booking_date'];
        $selectedBookingTime = $data['booking_time'];
        $bookedSlotDate = date(
            "d M, Y",
            strtotime($selectedBookingDate)
        )." ".$selectedBookingTime;

        if (empty($data['slot_day_index'])) {
            $data['parent_slot_id'] = 0;
            $data['slot_id'] = 0;
            $data['slot_day_index'] = 0;
        }
        $parentSlotId = $data['parent_slot_id'];
        $slotId = $data['slot_id'];
        $slotDayIndex = $data['slot_day_index'];

        // Check if selected booking dates are available or not
        $productId = $product->getId();
        $bookingInfo = $this->getBookingInfo($productId);
        $bookingSlotData = $this->getJsonDecodedString(
            $bookingInfo['info']
        );
        $slotData = [];
        $bookedData = $this->getBookedAppointmentDates($productId);
        if (empty($bookingSlotData[$slotDayIndex][$parentSlotId]['slots_info'][$slotId])) {
            $errorMessage = __('Invalid booking dates.');
            if ($isThrowError) {
                $this->_getSession->getQuote()->setHasError(true);
                throw new \Magento\Framework\Exception\LocalizedException(
                    $errorMessage
                );
            } else {
                $this->messageManager->addError($errorMessage);
            }
        } else {
            $slotData = $bookingSlotData[$slotDayIndex][$parentSlotId]['slots_info'][$slotId];

            $bookedDay = date("l", strtotime($selectedBookingDate));
            $bookedDayIndex = $this->getDayIndexId($bookedDay);
            if (!empty($bookingSlotData[$bookedDayIndex])) {
                // if selected time slot is available
                if (!empty($slotData['time'])) {
                    $selectedBookingTime = $slotData['time'];
                } else {
                    $error = 1;
                }
                if ($error) {
                    if ($isThrowError) {
                        $this->_getSession->getQuote()->setHasError(true);
                        throw new \Magento\Framework\Exception\LocalizedException($errorMessage);
                    } else {
                        $this->messageManager->addError($errorMessage);
                    }
                }

                // check if selected booking dates are for today then slot is available or not
                if (strtotime($selectedBookingDate)===strtotime(date('m/d/Y'))) {
                    if (!($currentTime <= strtotime($selectedBookingTime))) {
                        $error = 1;
                        $errorMessage = __(
                            'Slot %1 is not available.',
                            $bookedSlotDate
                        );
                    }
                }

                // get valid available booking dates
                $validBookingDates = $this->getValidBookingDates($product);
                $bookingAvailableFrom = $validBookingDates['booking_available_from'];
                $bookingAvailableTo = $validBookingDates['booking_available_to'];
                // check if selected booking dates are correct or not
                if (!(strtotime($bookingAvailableFrom)<=strtotime($selectedBookingDate))) {
                    $error = 1;
                    $errorMessage = __('Invalid booking dates.');
                }
                if (!$product['available_every_week']) {
                    if (!(strtotime($selectedBookingDate)<=strtotime($bookingAvailableTo))) {
                        $error = 1;
                        $errorMessage = __('Invalid booking dates.');
                    }
                }
            } else {
                $error = 1;
            }
            if ($error) {
                if ($isThrowError) {
                    $this->_getSession->getQuote()->setHasError(true);
                    throw new \Magento\Framework\Exception\LocalizedException($errorMessage);
                } else {
                    $this->messageManager->addError($errorMessage);
                }
            }
            $availableQty = 0;
            if (!empty($slotData['qty'])) {
                $availableQty = $slotData['qty'];
            }
            if (!empty($bookedData[strtotime($selectedBookingDate)][$selectedBookingTime])) {
                $bookedQty = $bookedData[strtotime($selectedBookingDate)][$selectedBookingTime];
                if ($bookedQty > $availableQty) {
                    $availableQty = 0;
                } else {
                    $availableQty = $availableQty - $bookedQty;
                }
            }
            $requestedQty = $item->getQty();
            if (!$availableQty) {
                $errorMessage = __(
                    '%1 quantity is not available for slot %2.',
                    $item->getName(),
                    $bookedSlotDate
                );

                $item->setHasError(true);
                $item->setMessage([$errorMessage]);
                if ($item->getId()) {
                    $item->delete();
                }
            } else {
                if ($requestedQty > $availableQty) {
                    $item->setQty($availableQty)->save();
                    $error = 1;
                    $errorMessage = __(
                        'Only %1 quantity is available for %2 for slot %3.',
                        $availableQty,
                        $item->getName(),
                        $bookedSlotDate
                    );
                    if ($isThrowError) {
                        $this->_getSession->getQuote()->setHasError(true);
                        throw new \Magento\Framework\Exception\LocalizedException(
                            $errorMessage
                        );
                    } else {
                        $this->messageManager->addError($errorMessage);
                        $this->_getSession->getQuote()->collectTotals()->save();
                    }
                }
                // save slot item data in booking quote table
                if ($itemId = $item->getId()) {
                    $collection = $this->quoteCollection->create();
                    $bookingQuote = $this->getDataByField($itemId, 'item_id', $collection);
                    if (!empty($item->getQuoteId())) {
                        $quoteId = $item->getQuoteId();
                    } else {
                        $quoteId = $this->_getSession->getQuote()->getId();
                    }
                    if (!$bookingQuote) {
                        $data =  [
                            'item_id' => $itemId,
                            'slot_id' => $slotId,
                            'parent_slot_id' => $parentSlotId,
                            'slot_day_index' => $slotDayIndex,
                            'slot_date' => $selectedBookingDate,
                            'slot_time' => $selectedBookingTime,
                            'quote_id' => $quoteId,
                            'product_id' => $productId
                        ];
                        $this->quote->create()->setData($data)->save();
                    }
                }
            }
        }
    }

    /**
     * Check Item Qty Avilable For Hotel Booking
     *
     * @param array $data
     * @param \Magento\Catalog\Model\Product $product
     * @param \Magento\Sales\Model\Order\Item $item
     * @param object $bookingQuote
     * @return void
     */
    public function checkItemQtyAvilableForHotel($data, $product, $item, $bookingQuote)
    {
        $result = $this->getBookedHotelDates($product);
        $error = false;
        $errorMessage = __("something went wrong !!!");

        if (empty($data['selected_configurable_option'])
            && empty($data['options'])
        ) {
            return null;
        }

        $selectedProductId = $data['selected_configurable_option'];
        // Check if selected booking options are available or not
        $bookingDateOptions = $this->getHotelBookingDateOptions($product);
        $bookingFromDate = $bookingToDate = 0;
        $actualAssociatedQty = $this->getStockData($selectedProductId)->getQty();

        if (!empty($bookingDateOptions)) {
            foreach ($bookingDateOptions as $optionId => $optionValues) {
                if ($optionValues['title'] == "Booking From") {
                    $bookingFromDate = $optionId;
                } elseif ($optionValues['title'] == "Booking Till") {
                    $bookingToDate = $optionId;
                }
            }
        }

        if (isset($data['options'][$bookingFromDate])
            && isset($data['options'][$bookingToDate])
        ) {
            $roomBookingFrom = $data['options'][$bookingFromDate];
            $roomBookingTo = $data['options'][$bookingToDate];
            $errorMessage = __(
                'Room(s) are not available during %1 to %2.',
                $roomBookingFrom,
                $roomBookingTo
            );
            $selectedBookingDateFrom = strtotime($roomBookingFrom);
            $selectedBookingDateTo = strtotime($roomBookingTo);

            if (isset($result[$selectedProductId]['booked_dates'])) {
                $_array = [];
                $bookedDatesArr = $result[$selectedProductId]['booked_dates'];
                foreach ($bookedDatesArr as $bookedDate => $qtyAvailable) {
                    $bookedDatesStr = strtotime($bookedDate);
                    if ($bookedDatesStr >= $selectedBookingDateFrom && $bookedDatesStr <= $selectedBookingDateTo) {
                        $_array[] = $qtyAvailable;
                    }
                }
                $itemId = (int) $item->getId();
                $qtyRequested = $item->getProduct()->getCartQty();
                $paramData = $this->getParams();
                if (!empty($paramData['item'][$itemId]['qty'])) {
                    $qtyRequested = $paramData['item'][$itemId]['qty'];
                }
                if (!empty($_array)) {
                    $actualQtyAvailable = min($_array);
                    if ($actualQtyAvailable!=="" && $actualQtyAvailable == 0) {
                        $error = true;
                    } elseif ($actualQtyAvailable
                        && $actualQtyAvailable > 0
                        && $qtyRequested > $actualQtyAvailable
                    ) {
                        $error = true;
                        if ($actualQtyAvailable > 0) {
                            $errorMessage = __(
                                'Only %1 Room(s) are available during %2 to %3.',
                                $actualQtyAvailable,
                                $roomBookingFrom,
                                $roomBookingTo
                            );
                        }
                    }
                }
            }

            if (!$error) {
                if ($bookingQuote->getSlotDate() && $bookingQuote->getToSlotDate()
                    && !(($actualAssociatedQty - $bookingQuote->getQty()) >= $data['qty'])
                ) {
                    $error = true;
                    if ($actualAssociatedQty - $bookingQuote->getQty() > 0) {
                        $errorMessage = __(
                            'Only %1 Room(s) are available during %2 to %3.',
                            $actualAssociatedQty - $bookingQuote->getQty(),
                            $roomBookingFrom,
                            $roomBookingTo
                        );
                    }
                }
            }
        }
        if ($error) {
            $this->_getSession->getQuote()->setHasError(true);
            throw new \Magento\Framework\Exception\LocalizedException(
                $errorMessage
            );
        }
    }

    /**
     * Check Hotel Is booked for selected Date range or not
     *
     * @param array $data
     * @param \Magento\Catalog\Model\Product $product
     * @param \Magento\Sales\Model\Order\Item $item
     * @return void
     */
    public function checkIsHotelBookedForDateRange($data, $product, $item)
    {
        $result = $this->getBookedHotelDates($product);
        $error = false;
        $errorMessage = __("something went wrong !!!");
        if (!empty($data['selected_configurable_option'])
            && !empty($data['options'])
        ) {
            // Check if selected booking options are available or not

            $bookingDateOptions = $this->getHotelBookingDateOptions($product);
            $bookingFromDate = $bookingToDate = 0;
            $selectedProductId = $data['selected_configurable_option'];
            $actualAssociatedQty = $this->getStockData($selectedProductId)->getQty();

            if (!empty($bookingDateOptions)) {
                foreach ($bookingDateOptions as $optionId => $optionValues) {
                    if ($optionValues['title'] == "Booking From") {
                        $bookingFromDate = $optionId;
                    } elseif ($optionValues['title'] == "Booking Till") {
                        $bookingToDate = $optionId;
                    }
                }
            }

            if (isset($data['options'][$bookingFromDate]) && isset($data['options'][$bookingToDate])) {
                $errorMessage = __(
                    'Room(s) are not available during %1 to %2.',
                    $data['options'][$bookingFromDate],
                    $data['options'][$bookingToDate]
                );
                $selectedBookingDateFrom = strtotime($data['options'][$bookingFromDate]);
                $selectedBookingDateTo = strtotime($data['options'][$bookingToDate]);

                if (isset($result[$selectedProductId]['booked_dates'])) {
                    $_array = [];
                    foreach ($result[$selectedProductId]['booked_dates'] as $bookedDate => $qtyAvailable) {
                        $bookedDatesStr = strtotime($bookedDate);
                        if ($bookedDatesStr >= $selectedBookingDateFrom && $bookedDatesStr <= $selectedBookingDateTo) {
                            $_array[] = $qtyAvailable;
                        }
                    }
                    $actualQtyAvailable = '';
                    if (!empty($_array)) {
                        $actualQtyAvailable = min($_array);
                    }

                    $qtyRequested = $item->getProduct()->getCartQty();
                    $itemId = $item->getId();
                    $paramData = $this->getParams();
                    if (!empty($paramData['item'][$itemId]['qty'])) {
                        $qtyRequested = $paramData['item'][$itemId]['qty'];
                    }

                    if (!empty($_array) && $actualQtyAvailable!=="" && $actualQtyAvailable == 0) {
                        $error = true;
                    } elseif (!empty($_array)
                        && $actualQtyAvailable
                        && $actualQtyAvailable > 0
                        && $qtyRequested > $actualQtyAvailable
                    ) {
                        $error = true;
                        if ($actualQtyAvailable > 0) {
                            $errorMessage = __(
                                'Only %1 Room(s) are available during %2 to %3.',
                                $actualQtyAvailable,
                                $data['options'][$bookingFromDate],
                                $data['options'][$bookingToDate]
                            );
                        }
                    }
                }

                if ($error) {
                    $this->_getSession->getQuote()->setHasError(true);
                    throw new \Magento\Framework\Exception\LocalizedException(
                        $errorMessage
                    );
                }

                $items =  $this->_getSession->getQuote()->getAllVisibleItems();
                foreach ($items as $tempItem) {
                    if ($tempItem->getProductType() != "hotelbooking" && !$tempItem->getId()) {
                        continue;
                    }

                    $itemData = $tempItem->getBuyRequest()->getData();
                    $itemId = (int) $tempItem->getId();
                    $paramData = $this->getParams();
                    if (!empty($paramData['item'][$itemId]['selected_configurable_option'])) {
                        $itemData = $paramData['item'][$itemId];
                        $tempSelectedProductId = $itemData['selected_configurable_option'];
                    } else {
                        $tempSelectedProductId = $data['selected_configurable_option'];
                    }

                    if ($tempSelectedProductId == $selectedProductId
                        && isset($itemData['options'][$bookingFromDate])
                        && isset($itemData['options'][$bookingToDate])
                    ) {
                        $tempSelectedBookingDateFrom = strtotime($itemData['options'][$bookingFromDate]);
                        $tempSelectedBookingDateTo = strtotime($itemData['options'][$bookingToDate]);

                        $remainingQty = $actualAssociatedQty - $tempItem->getQty();

                        $qtyRequested = $item->getProduct()->getCartQty();
                        $paramData = $this->getParams();
                        if (!empty($paramData['item'][$itemId]['qty'])) {
                            $qtyRequested = $paramData['item'][$itemId]['qty'];
                        }

                        if ($selectedBookingDateTo < $tempSelectedBookingDateFrom
                            || $selectedBookingDateFrom > $tempSelectedBookingDateTo
                        ) {
                            $error = false;
                        } elseif ($remainingQty >= $qtyRequested) {
                            $error = false;
                        } else {
                            $error = true;
                        }
                    }
                }
            }
        }
        if ($error) {
            $this->_getSession->getQuote()->setHasError(true);
            throw new \Magento\Framework\Exception\LocalizedException(
                $errorMessage
            );
        }
    }

    /**
     * Process Hotel Booking Save Data
     *
     * @param array                           $data
     * @param \Magento\Catalog\Model\Product  $product
     * @param \Magento\Sales\Model\Order\Item $item
     * @param int                             $isThrowError
     * @return void
     */
    public function processHotelBookingSave($data, $product, $item, $isThrowError = 1)
    {
        if (!empty($data['selected_configurable_option'])
            && !empty($data['options'])
        ) {
            // if (isset($data['hotel_qty'])) {
            //     $price = $item->getProduct()->getFinalPrice();
            //     if ($data['hotel_qty'] > 1) {
            //         $item->setCustomPrice($price*$data['hotel_qty']);
            //         $item->setOriginalCustomPrice($price*$data['hotel_qty']);
            //     } else {
            //         $item->setCustomPrice($price);
            //         $item->setOriginalCustomPrice($price);
            //     }
            // }

            // Check if selected booking options are available or not
            $error = 0;
            $errorMessage = __('Invalid booking dates.');
            $productId = $product->getId();
            $bookingDateOptions = $this->getHotelBookingDateOptions($product);
            $bookingFromDate = $bookingToDate = 0;

            if (!empty($bookingDateOptions)) {
                foreach ($bookingDateOptions as $optionId => $optionValues) {
                    if ($optionValues['title'] == "Booking From") {
                        $bookingFromDate = $optionId;
                    } elseif ($optionValues['title'] == "Booking Till") {
                        $bookingToDate = $optionId;
                    }
                }
            }

            if (isset($data['options'][$bookingFromDate]) && isset($data['options'][$bookingToDate])) {
                $selectedBookingDateFrom = $data['options'][$bookingFromDate];
                $selectedBookingDateTo = $data['options'][$bookingToDate];
                if (!$error) {
                    // get valid available booking dates
                    $bookingAvailableFrom = $this->getCurrentDate();
                    // check if selected booking dates are correct or not
                    if (!(strtotime($bookingAvailableFrom)<=strtotime($selectedBookingDateFrom))) {
                        $error = 1;
                        $errorMessage = __('Invalid booking dates.');
                        if ($isThrowError) {
                            $this->_getSession->getQuote()->setHasError(true);
                            throw new \Magento\Framework\Exception\LocalizedException(
                                $errorMessage
                            );
                        } else {
                            $this->messageManager->addError($errorMessage);
                        }
                    }
                } else {
                    $error = 1;
                }
                if ($error) {
                    if ($isThrowError) {
                        $this->_getSession->getQuote()->setHasError(true);
                        throw new \Magento\Framework\Exception\LocalizedException($errorMessage);
                    } else {
                        $this->messageManager->addError($errorMessage);
                    }
                }
                // save slot item data in booking quote table
                if ($itemId = $item->getId()) {
                    $collection = $this->quoteCollection->create();
                    $bookingQuote = $this->getDataByField($itemId, 'item_id', $collection);
                    if (!empty($item->getQuoteId())) {
                        $quoteId = $item->getQuoteId();
                    } else {
                        $quoteId = $this->_getSession->getQuote()->getId();
                    }
                    if (!$bookingQuote) {
                        $data =  [
                            'item_id' => $itemId,
                            'slot_id' => 0,
                            'parent_slot_id' => 0,
                            'slot_day_index' => 0,
                            'slot_date' => $selectedBookingDateFrom,
                            'to_slot_date' => $selectedBookingDateTo,
                            'quote_id' => $quoteId,
                            'qty' => $item->getQty(),
                            'product_id' => $productId
                        ];
                        $this->quote->create()->setData($data)->save();
                    }
                }
            }
        }
    }

    /**
     * Process Event Booking Save Data
     *
     * @param array $data
     * @param \Magento\Catalog\Model\Product $product
     * @param \Magento\Sales\Model\Order\Item $item
     * @param int $isThrowError
     * @return void
     */
    public function processEventBookingSave($data, $product, $item, $isThrowError = 1)
    {
        if (empty($data['options'])) {
            return null;
        }

        $errorMessage = __('Invalid Tickets.');
        // Check if selected booking options are available or not
        $productId = $product->getId();
        $bookingInfo = $this->getBookingInfo($productId);
        $eventOptionsData = [];
        $eventOptions = $this->getEventOptions($product);
        if (!empty($eventOptions['event_ticket'])) {
            $eventOptionsData = $eventOptions['event_ticket'];
        }
        $eventDateStart = date(
            'Y-m-d',
            strtotime($bookingInfo['start_date'])
        );
        $eventTimeStart =  date(
            'h:i a',
            strtotime($bookingInfo['start_date'])
        );
        $eventDateEnd = date(
            'Y-m-d',
            strtotime($bookingInfo['end_date'])
        );
        $eventTimeEnd =  date(
            'h:i a',
            strtotime($bookingInfo['end_date'])
        );
        $savedOptionId = 0;
        // get saved event option id
        if (!empty($eventOptionsData['option_id'])) {
            $savedOptionId = $eventOptionsData['option_id'];
        }

        if (empty($data['options'][$savedOptionId])) {
            if ($isThrowError) {
                $item->setHasError(true);
                $item->setMessage([$errorMessage]);
                throw new \Magento\Framework\Exception\LocalizedException(
                    $errorMessage
                );
            } else {
                $this->messageManager->addError($errorMessage);
                $item->setHasError(true);
                $item->setMessage([$errorMessage]);
            }

            return null;
        }

        // if product is added with saved event option id
        // if product option have only one ticket value
        // if (count($data['options'][$savedOptionId]) != 1) {
        //     return null;
        // }

        $optionValId = $data['options'][$savedOptionId][0];
        $savedOptionValues = [];
        $savedOptionQty = [];
        $savedOptionTitle = [];
        $savedOptionInStock = [];

        // check for booked data
        $bookedData = $this->getBookedEventData($productId, $bookingInfo, $savedOptionId, $optionValId);

        foreach ($eventOptionsData['option_values'] as $key => $value) {
            if (empty($value['option_type_id'])) {
                break;
            }
            array_push($savedOptionValues, $value['option_type_id']);
            $savedOptionQty[$value['option_type_id']] = $value['qty'];
            $savedOptionTitle[$value['option_type_id']] = $value['title'];
            $savedOptionInStock[$value['option_type_id']] = $value['is_in_stock'];
        }
        // if product option ticket value is exist or not
        if (!in_array($optionValId, $savedOptionValues)) {
            $errorMessage = __('Tickets is not available.');
            if ($isThrowError) {
                $item->setHasError(true);
                $item->setMessage([$errorMessage]);
                throw new \Magento\Framework\Exception\LocalizedException(
                    $errorMessage
                );
            } else {
                $item->setHasError(true);
                $item->setMessage([$errorMessage]);
                $this->messageManager->addError($errorMessage);
            }
        }
        if (!empty($savedOptionQty[$optionValId])) {
            $availableQty = $savedOptionQty[$optionValId];
        } else {
            $availableQty = 0;
        }

        if (!empty($bookedData[$savedOptionId][$optionValId])) {
            $bookedQty = $bookedData[$savedOptionId][$optionValId];
            if ($bookedQty > $availableQty) {
                $availableQty = 0;
            } else {
                $availableQty = $availableQty - $bookedQty;
            }
        }
        if (empty($savedOptionInStock[$optionValId]) || !$availableQty) {
            if (isset($savedOptionTitle[$optionValId])) {
                $errorMessage = __(
                    '"%1" ticket for "%2" is out of stock.',
                    $savedOptionTitle[$optionValId],
                    $item->getName()
                );
            } else {
                $errorMessage = __(
                    'Ticket for "%1" is out of stock.',
                    $item->getName()
                );
            }
            $item->setHasError(true);
            $item->setMessage([$errorMessage]);
            if ($item->getId()) {
                $item->delete();
            }
        } else {
            if ($item->getQty()>$availableQty) {
                $item->setQty($availableQty)->save();
                if ($item->getId()) {
                    $item->getQuote()->collectTotals();
                }
                if (empty($savedOptionTitle[$optionValId])) {
                    $savedOptionTitle[$optionValId] = '';
                }
                $errorMessage = __(
                    'Only %1 quantity is available for %2 ticket "%3".',
                    $availableQty,
                    $item->getName(),
                    $savedOptionTitle[$optionValId]
                );
                if ($isThrowError) {
                    $item->setHasError(true);
                    $item->setMessage([$errorMessage]);
                    throw new \Magento\Framework\Exception\LocalizedException(
                        $errorMessage
                    );
                } else {
                    $this->messageManager->addError($errorMessage);
                    $item->setHasError(true);
                    $item->setMessage([$errorMessage]);
                }
            } else {
                if (!$item->getId() || !$item->getQty()) {
                    if ($data['original_qty'] > $availableQty) {
                        if (empty($savedOptionTitle[$optionValId])) {
                            $savedOptionTitle[$optionValId] = '';
                        }
                        $errorMessage = __(
                            'Only %1 quantity is available for %2 ticket "%3".',
                            $availableQty,
                            $item->getName(),
                            $savedOptionTitle[$optionValId]
                        );
                        
                        if ($isThrowError) {
                            $item->setHasError(true);
                            $item->setMessage([$errorMessage]);
                            throw new \Magento\Framework\Exception\LocalizedException(
                                $errorMessage
                            );
                        } else {
                            $this->messageManager->addError($errorMessage);
                            $item->setHasError(true);
                            $item->setMessage([$errorMessage]);
                        }
                    }
                }
            }
            // save slot item data in booking quote table
            if ($itemId = $item->getId()) {
                $collection = $this->quoteCollection->create();
                $bookingQuote = $this->getDataByField($itemId, 'item_id', $collection);
                if (!empty($item->getQuoteId())) {
                    $quoteId = $item->getQuoteId();
                } else {
                    $quoteId = $this->_getSession->getQuote()->getId();
                }
                if (!$bookingQuote) {
                    $data =  [
                        'item_id' => $itemId,
                        'slot_id' => $optionValId,
                        'parent_slot_id' => $savedOptionId,
                        'slot_day_index' => '',
                        'slot_date' => $eventDateStart,
                        'slot_time' => $eventTimeStart,
                        'to_slot_date' => $eventDateEnd,
                        'to_slot_time' => $eventTimeEnd,
                        'quote_id' => $quoteId,
                        'qty' => $item->getQty(),
                        'product_id' => $productId
                    ];
                    $this->quote->create()->setData($data)->save();
                }
            }
        }
    }

    /**
     * Process Rent Booking Save
     *
     * @param array $data
     * @param \Magento\Catalog\Model\Product $product
     * @param \Magento\Sales\Model\Order\Item $item
     * @param int $isThrowError
     * @return void
     */
    public function processRentBookingSave($data, $product, $item, $isThrowError = 1)
    {
        if (empty($data['options']) || empty($data['booking_date_from'])) {
            return null;
        }
        
        $rentOptions = $this->getRentOptions($product);
        $currentTime = $this->getCurrentTime();
        $rentFromOptId = '';
        if (!empty($rentOptions['rent_from']['option_id'])) {
            $rentFromOptId = $rentOptions['rent_from']['option_id'];
        }
        $rentToOptId = '';
        if (!empty($rentOptions['rent_to']['option_id'])) {
            $rentToOptId = $rentOptions['rent_to']['option_id'];
        }
        $fromDateFromated = '';
        if (!empty($data['options'][$rentFromOptId])) {
            $fromDateFromated = $data['options'][$rentFromOptId];
        }
        $toDateFromated = '';
        if (!empty($data['options'][$rentToOptId])) {
            $toDateFromated = $data['options'][$rentToOptId];
        }
        $error = 0;
        $errorMessage = __('Invalid booking dates.');
        if (empty($data['slot_day_index'])) {
            $data['parent_slot_id'] = 0;
            $data['slot_id'] = 0;
            $data['slot_day_index'] = 0;
            $data['booking_from_time'] = 0;
            $data['booking_to_time'] = 0;
        }
        $parentSlotId = $data['parent_slot_id'];
        $slotId = $data['slot_id'];
        $slotIdFrom = $data['booking_from_time'];
        $slotIdTo = $data['booking_to_time'];
        $slotDayIndex = $data['slot_day_index'];

        // Check if selected booking dates are available or not
        $productId = $product->getId();
        $bookingInfo = $this->getBookingInfo($productId);
        $availableSavedQty = $bookingInfo['available_qty'];
        $bookingSlotData = $this->getJsonDecodedString(
            $bookingInfo['info']
        );
        $isSlotExisted = 0;
        $slotDataFrom = [];
        $slotDataTo = [];
        $rentType = Info::RENT_TYPE_DAILY;
        if ($slotDayIndex) {
            $rentType = Info::RENT_TYPE_HOURLY;
            if (!empty($bookingSlotData[$slotDayIndex][$parentSlotId]['slots_info'][$slotIdFrom])
                && !empty($bookingSlotData[$slotDayIndex][$parentSlotId]['slots_info'][$slotIdTo])
            ) {
                $isSlotExisted = 1;
            }
        }

        $bookedSlotFromDate = $data['booking_date_from'];
        if ($rentType == Info::RENT_TYPE_HOURLY) {
            $bookedSlotToDate = $data['booking_date_from'];
            $selectedBookingFromTime = '';
            $selectedBookingToTime = '';
        } else {
            $bookedSlotToDate = $data['booking_date_to'];
            $selectedBookingFromTime = date(
                "h:i a",
                strtotime($bookedSlotFromDate)
            );
            $selectedBookingToTime = '11:59 pm';
        }
        $selectedBookingFromDate = date(
            "Y-m-d",
            strtotime($bookedSlotFromDate)
        );
        $selectedBookingToDate = date(
            "Y-m-d",
            strtotime($bookedSlotToDate)
        );

        if ($rentType == Info::RENT_TYPE_HOURLY && !$isSlotExisted) {
            $errorMessage = __('Invalid booking dates.');
            if ($isThrowError) {
                $this->_getSession->getQuote()->setHasError(true);
                throw new \Magento\Framework\Exception\LocalizedException(
                    $errorMessage
                );
            } else {
                $this->messageManager->addError($errorMessage);
            }
        } elseif ($rentType == Info::RENT_TYPE_DAILY) {
            // get valid available booking dates
            $validBookingDates = $this->getValidBookingDates($product);
            $bookingAvailableFrom = $validBookingDates['booking_available_from'];
            $bookingAvailableTo = $validBookingDates['booking_available_to'];
            // check if selected booking dates are correct or not
            if (!(strtotime($bookingAvailableFrom)<=strtotime($bookedSlotFromDate))) {
                $error = 1;
                $errorMessage = __('Invalid booking dates.');
                if ($isThrowError) {
                    $this->_getSession->getQuote()->setHasError(true);
                    throw new \Magento\Framework\Exception\LocalizedException(
                        $errorMessage
                    );
                } else {
                    $this->messageManager->addError($errorMessage);
                }
            }
            if (!$product['available_every_week']) {
                if (!(strtotime($bookedSlotToDate)<=strtotime($bookingAvailableTo))) {
                    $error = 1;
                    $errorMessage = __('Invalid booking dates.');
                    if ($isThrowError) {
                        $this->_getSession->getQuote()->setHasError(true);
                        throw new \Magento\Framework\Exception\LocalizedException(
                            $errorMessage
                        );
                    } else {
                        $this->messageManager->addError($errorMessage);
                    }
                }
            }
            $totalOrderedQty = $this->getTotalOrderedRentedQty(
                $productId,
                $item->getId(),
                $selectedBookingFromDate,
                $selectedBookingToDate
            );
            if ($totalOrderedQty > $availableSavedQty) {
                $totalOrderedQty = $availableSavedQty;
            }
            $availableSavedQty = $availableSavedQty - $totalOrderedQty;
            if ($availableSavedQty <= 0) {
                $errorMessage = __(
                    '%1 is not available for dates %2 to %3.',
                    $item->getName(),
                    $fromDateFromated,
                    $toDateFromated
                );

                $item->setHasError(true);
                $item->setMessage([$errorMessage]);
                if ($item->getId()) {
                    $item->delete();
                }
            } else {
                $requestedQty = $item->getQty();

                if ($requestedQty > $availableSavedQty) {
                    $item->setQty($availableSavedQty)->save();
                    $this->_getSession->getQuote()->save();

                    $error = 1;
                    $errorMessage = __(
                        'Only %1 quantity is available for %2 for dates %3 to %4.',
                        $availableSavedQty,
                        $item->getName(),
                        $fromDateFromated,
                        $toDateFromated
                    );
                    if ($isThrowError) {
                        $this->_getSession->getQuote()->setHasError(true);
                        throw new \Magento\Framework\Exception\LocalizedException(
                            $errorMessage
                        );
                    } else {
                        $this->messageManager->addError($errorMessage);
                    }
                    if ($item->getId()) {
                        //to update cart item data in booking quote table
                        $collection = $this->quoteCollection->create();
                        $bookingQuote = $this->getDataByField($item->getId(), 'item_id', $collection);
                        if (!empty($item->getQuoteId())) {
                            $quoteId = $item->getQuoteId();
                        } else {
                            $quoteId = $this->_getSession->getQuote()->getId();
                        }
                        if (!$bookingQuote) {
                            $data =  [
                                'item_id' => $item->getId(),
                                'slot_id' => $slotId,
                                'parent_slot_id' => $parentSlotId,
                                'slot_day_index' => $slotDayIndex,
                                'slot_date' => $selectedBookingFromDate,
                                'slot_time' => $selectedBookingFromTime,
                                'to_slot_date' => $selectedBookingToDate,
                                'to_slot_time' => $selectedBookingToTime,
                                'rent_type' => $rentType,
                                'quote_id' => $quoteId,
                                'qty' => $item->getQty(),
                                'product_id' => $productId
                            ];
                            $this->quote->create()->setData($data)->save();
                        }
                    } else {
                        $this->_getSession->getQuote()->removeItem($item->getId())->save();
                    }
                }
                // save slot item data in booking quote table
                if ($error !==1 && $itemId = $item->getId()) {
                    $collection = $this->quoteCollection->create();
                    $bookingQuote = $this->getDataByField($itemId, 'item_id', $collection);
                    if (!empty($item->getQuoteId())) {
                        $quoteId = $item->getQuoteId();
                    } else {
                        $quoteId = $this->_getSession->getQuote()->getId();
                    }
                    if (!$bookingQuote) {
                        $data =  [
                            'item_id' => $itemId,
                            'slot_id' => 0,
                            'parent_slot_id' => 0,
                            'slot_day_index' => 0,
                            'slot_date' => $selectedBookingFromDate,
                            'slot_time' => $selectedBookingFromTime,
                            'to_slot_date' => $selectedBookingToDate,
                            'to_slot_time' => $selectedBookingToTime,
                            'rent_type' => $rentType,
                            'quote_id' => $quoteId,
                            'qty' => $item->getQty(),
                            'product_id' => $productId
                        ];
                        $this->quote->create()->setData($data)->save();
                    }
                }
            }
        } elseif ($rentType == Info::RENT_TYPE_HOURLY && $isSlotExisted) {
            $bookingSlotDataArr = $bookingSlotData[$slotDayIndex][$parentSlotId];
            $slotDataFrom = $bookingSlotDataArr['slots_info'][$slotIdFrom];
            $slotDataTo = $bookingSlotDataArr['slots_info'][$slotIdTo];

            $bookedDay = date("l", strtotime($selectedBookingFromDate));
            $bookedDayIndex = $this->getDayIndexId($bookedDay);
            if (!empty($bookingSlotData[$bookedDayIndex])) {
                // if selected time slot is available
                if (empty($slotDataFrom['time']) && empty($slotDataTo['time'])) {
                    $error = 1;
                } else {
                    $selectedBookingFromTime = $slotDataFrom['time'];
                    $selectedBookingToTime = $slotDataTo['time'];
                }
                if ($error) {
                    if ($isThrowError) {
                        $this->_getSession->getQuote()->setHasError(true);
                        throw new \Magento\Framework\Exception\LocalizedException($errorMessage);
                    } else {
                        $this->messageManager->addError($errorMessage);
                    }
                }

                // check if selected booking dates are for today then slot is available or not
                if (strtotime($selectedBookingFromDate)===strtotime(date('m/d/Y'))) {
                    if (!($currentTime <= strtotime($selectedBookingFromTime))) {
                        $error = 1;
                        $errorMessage = __(
                            'Rent from %1 to %2 is not available.',
                            $fromDateFromated,
                            $toDateFromated
                        );
                        if ($isThrowError) {
                            $this->_getSession->getQuote()->setHasError(true);
                            throw new \Magento\Framework\Exception\LocalizedException(
                                $errorMessage
                            );
                        } else {
                            $this->messageManager->addError($errorMessage);
                        }
                    }
                }
                if ($error) {
                    if ($isThrowError) {
                        $this->_getSession->getQuote()->setHasError(true);
                        throw new \Magento\Framework\Exception\LocalizedException($errorMessage);
                    } else {
                        $this->messageManager->addError($errorMessage);
                    }
                }
                
                // get valid available booking dates
                $validBookingDates = $this->getValidBookingDates($product);
                $bookingAvailableFrom = $validBookingDates['booking_available_from'];
                $bookingAvailableTo = $validBookingDates['booking_available_to'];
                // check if selected booking dates are correct or not
                if (!(strtotime($bookingAvailableFrom)<=strtotime($bookedSlotFromDate))) {
                    $error = 1;
                    $errorMessage = __('Invalid booking dates.');
                    if ($isThrowError) {
                        $this->_getSession->getQuote()->setHasError(true);
                        throw new \Magento\Framework\Exception\LocalizedException(
                            $errorMessage
                        );
                    } else {
                        $this->messageManager->addError($errorMessage);
                    }
                }
                if (!$product['available_every_week']) {
                    if (!(strtotime($selectedBookingToDate)<=strtotime($bookingAvailableTo))) {
                        $error = 1;
                        $errorMessage = __('Invalid booking dates.');
                        if ($isThrowError) {
                            $this->_getSession->getQuote()->setHasError(true);
                            throw new \Magento\Framework\Exception\LocalizedException(
                                $errorMessage
                            );
                        } else {
                            $this->messageManager->addError($errorMessage);
                        }
                    }
                }

                $totalOrderedQty = $this->getTotalOrderedRentedQty(
                    $productId,
                    $item->getId(),
                    $selectedBookingFromDate,
                    $selectedBookingToDate
                );
                if ($totalOrderedQty > $availableSavedQty) {
                    $totalOrderedQty = $availableSavedQty;
                }
                $availableSavedQty = $availableSavedQty - $totalOrderedQty;
                if ($availableSavedQty <= 0) {
                    $errorMessage = __(
                        '%1 is not available from %2 to %3.',
                        $item->getName(),
                        $fromDateFromated,
                        $toDateFromated
                    );

                    $item->setHasError(true);
                    $item->setMessage([$errorMessage]);
                    if ($item->getId()) {
                        $item->delete();
                    }
                    // $error = 1;
                    return;
                }
            } else {
                $error = 1;
            }
            if ($error) {
                if ($isThrowError) {
                    $this->_getSession->getQuote()->setHasError(true);
                    throw new \Magento\Framework\Exception\LocalizedException($errorMessage);
                } else {
                    $this->messageManager->addError($errorMessage);
                }
            }
            $availableFromQty = 0;
            $availableToQty = 0;
            $availableQty = 0;
            if (!empty($slotDataFrom['qty']) && !empty($slotDataTo['qty'])) {
                $availableFromQty = $slotDataFrom['qty'];
                $availableToQty = $slotDataTo['qty'];
                $qtyDiff = $availableToQty - $availableFromQty;
                if ($qtyDiff) {
                    $availableQty = $availableFromQty;
                } else {
                    $availableQty = $availableToQty;
                }
            }
            if ($availableQty <= 0) {
                $errorMessage = __(
                    '%1 is not available for dates %2 to %3.',
                    $item->getName(),
                    $fromDateFromated,
                    $toDateFromated
                );

                $item->setHasError(true);
                $item->setMessage([$errorMessage]);
                if ($item->getId()) {
                    $item->delete();
                }
            } else {
                $requestedQty = $item->getQty();
                if ($requestedQty > $availableFromQty || $requestedQty > $availableToQty) {
                    $item->setQty($availableQty)->save();
                    $this->_getSession->getQuote()->save();

                    $error = 1;
                    $errorMessage = __(
                        'Only %1 quantity is available for %2 for dates %3 to %4.',
                        $availableQty,
                        $item->getName(),
                        $fromDateFromated,
                        $toDateFromated
                    );
                    if ($isThrowError) {
                        $this->_getSession->getQuote()->setHasError(true);
                        throw new \Magento\Framework\Exception\LocalizedException(
                            $errorMessage
                        );
                    } else {
                        $this->messageManager->addError($errorMessage);
                        $this->_getSession->getQuote()->collectTotals();
                    }
                }
                // save slot item data in booking quote table
                if ($itemId = $item->getId()) {
                    $collection = $this->quoteCollection->create();
                    $bookingQuote = $this->getDataByField($itemId, 'item_id', $collection);
                    if (!empty($item->getQuoteId())) {
                        $quoteId = $item->getQuoteId();
                    } else {
                        $quoteId = $this->_getSession->getQuote()->getId();
                    }
                    if (!$bookingQuote) {
                        $data =  [
                            'item_id' => $itemId,
                            'slot_id' => $slotId,
                            'parent_slot_id' => $parentSlotId,
                            'slot_day_index' => $slotDayIndex,
                            'slot_date' => $selectedBookingFromDate,
                            'slot_time' => $selectedBookingFromTime,
                            'to_slot_date' => $selectedBookingToDate,
                            'to_slot_time' => $selectedBookingToTime,
                            'rent_type' => $rentType,
                            'quote_id' => $quoteId,
                            'qty' => $item->getQty(),
                            'product_id' => $productId
                        ];
                        $this->quote->create()->setData($data)->save();
                    }
                }
            }
        }
    }

    /**
     * Process Deafult Type Booking Slot Data
     *
     * @param array $data
     * @param \Magento\Sales\Model\Order\Item $item
     * @param int $productId
     * @param int $itemId
     * @param int $isThrowError
     * @return void
     */
    public function processDefaultSlotData($data, $item, $productId, $itemId, $isThrowError = 1)
    {
        $parentId = $this->getParentSlotId($productId);
        $slotId = (int) $data['slot_id'];
        $result = $this->processSlotData($data, $productId);
        if ($result['error']) {
            $this->messageManager->addNotice(__($result['msg']));
            if ($isThrowError) {
                $this->_getSession->getQuote()->setHasError(true);
                throw new \Magento\Framework\Exception\LocalizedException(__($result['msg']));
            }
        } else {
            if ($itemId > 0) {
                $collection = $this->quoteCollection->create();
                $tempitem = $this->getDataByField($itemId, 'item_id', $collection);

                if (!$tempitem) {
                    $data =  [
                        'item_id' => $itemId,
                        'slot_id' => $slotId,
                        'parent_slot_id' => $parentId,
                        'quote_id' => $item->getQuoteId(),
                        'product_id' => $productId
                    ];
                    $this->quote->create()->setData($data)->save();
                }
            }
        }
    }

    /**
     * Return Customer id.
     *
     * @return bool|0|1
     */
    public function getCustomerId()
    {
        return $this->_getSession->getCustomerId();
    }

    /**
     * GetCovertedPrice
     *
     * @param float $price
     * @return float
     */
    public function getCovertedPrice($price)
    {
        $price = $this->priceCurrency->convert(
            $price,
            $this->_getSession->getQuote()->getStore()
        );
        return $price;
    }

    /**
     * Process Slot Data for Table type Booking
     *
     * @param array $data
     * @param \Magento\Catalog\Model\Product $product
     * @param \Magento\Sales\Model\Order\Item $item
     * @param int $isThrowError
     */
    public function processTableBookingSave($data, $product, $item, $isThrowError = 1)
    {
        if (empty($data['booking_date']) && empty($data['booking_time'])) {
            return null;
        }
        $error = 0;
        $currentTime = $this->getCurrentTime();
        $errorMessage = __('Invalid booking dates.');
        $selectedBookingDate = $data['booking_date'];
        $selectedBookingTime = $data['booking_time'];
        $bookedSlotDate = date(
            "d M, Y",
            strtotime($selectedBookingDate)
        )." ".trim($selectedBookingTime);
        if (empty($data['slot_day_index'])) {
            $data['slot_day_index'] = 0;
        }
        if (empty($data['slot_id'])) {
            $data['slot_id'] = 0;
        }
        if (empty($data['parent_slot_id'])) {
            $data['parent_slot_id'] = 0;
        }
        $parentSlotId = $data['parent_slot_id'];
        $slotId = $data['slot_id'];
        $slotDayIndex = $data['slot_day_index'];

        // Check if selected booking dates are available or not
        $productId = $product->getId();
        $bookingInfo = $this->getBookingInfo($productId);
        $bookingSlotData = $this->getJsonDecodedString(
            $bookingInfo['info']
        );
        $slotData = [];
        $bookedData = $this->getBookedAppointmentDates($productId);
        if (empty($bookingSlotData[$slotDayIndex][$parentSlotId]['slots_info'][$slotId])) {
            $errorMessage = __('Invalid booking dates.');
            if ($isThrowError) {
                $this->_getSession->getQuote()->setHasError(true);
                throw new \Magento\Framework\Exception\LocalizedException(
                    $errorMessage
                );
            } else {
                $this->messageManager->addError($errorMessage);
            }
        } else {
            $slotData = $bookingSlotData[$slotDayIndex][$parentSlotId]['slots_info'][$slotId];
            $bookedDay = date("l", strtotime($selectedBookingDate));
            $bookedDayIndex = $this->getDayIndexId($bookedDay);

            if (!empty($bookingSlotData[$bookedDayIndex])) {
                // if selected time slot is available
                if (!empty($slotData['time'])) {
                    $selectedBookingTime = $slotData['time'];
                } else {
                    $error = 1;
                }
                if ($error) {
                    if ($isThrowError) {
                        $item->setHasError(true);
                        $item->setMessage([$errorMessage]);
                        $this->_getSession->getQuote()->setHasError(true);
                        throw new \Magento\Framework\Exception\LocalizedException($errorMessage);
                    } else {
                        $item->setHasError(true);
                        $item->setMessage([$errorMessage]);
                    }
                }
                // check if selected booking dates are for today then slot is available or not
                if (strtotime($selectedBookingDate)===strtotime(date('m/d/Y'))) {
                    if (!($currentTime <= strtotime($selectedBookingTime))) {
                        $error = 1;
                        $errorMessage = __(
                            'Slot %1 is not available.',
                            $bookedSlotDate
                        );
                        if ($isThrowError) {
                            $item->setHasError(true);
                            $item->setMessage([$errorMessage]);
                            $this->_getSession->getQuote()->setHasError(true);
                            throw new \Magento\Framework\Exception\LocalizedException(
                                $errorMessage
                            );
                        } else {
                            $item->setHasError(true);
                            $item->setMessage([$errorMessage]);
                        }
                    }
                }
                if (!$error) {
                    // get valid available booking dates
                    $bookingAvailableFrom = date('m/d/Y');
                    $bookingAvailableTo = '';

                    // check if selected booking dates are correct or not
                    if (!(strtotime($bookingAvailableFrom)<=strtotime($selectedBookingDate))) {
                        $error = 1;
                        $errorMessage = __('Invalid booking dates.');
                        if ($isThrowError) {
                            $item->setHasError(true);
                            $item->setMessage([$errorMessage]);
                            $this->_getSession->getQuote()->setHasError(true);
                            throw new \Magento\Framework\Exception\LocalizedException(
                                $errorMessage
                            );
                        } else {
                            $item->setHasError(true);
                            $item->setMessage([$errorMessage]);
                        }
                    }
                }
            } else {
                $error = 1;
            }
            if ($error) {
                if ($isThrowError) {
                    $item->setHasError(true);
                    $item->setMessage([$errorMessage]);
                    $this->_getSession->getQuote()->setHasError(true);
                    throw new \Magento\Framework\Exception\LocalizedException($errorMessage);
                } else {
                    $item->setHasError(true);
                    $item->setMessage([$errorMessage]);
                }
            }
            $availableQty = 0;
            if (!empty($slotData['qty'])) {
                $availableQty = $slotData['qty'];
            }
            if (!empty($bookedData[strtotime($selectedBookingDate)][$selectedBookingTime])) {
                $bookedQty = $bookedData[strtotime($selectedBookingDate)][$selectedBookingTime];
                if ($bookedQty > $availableQty) {
                    $availableQty = 0;
                } else {
                    $availableQty = $availableQty - $bookedQty;
                }
            }
            $requestedQty = $item->getQty();
            if (!empty($data['charged_per_count']) && $data['charged_per_count'] > 1) {
                $requestedQty = $requestedQty * $data['charged_per_count'];
            }
            if (!$availableQty) {
                $errorMessage = __(
                    '%1 quantity is not available for slot %2.',
                    $item->getName(),
                    $bookedSlotDate
                );

                $item->setHasError(true);
                $item->setMessage([$errorMessage]);
                if ($item->getId()) {
                    $item->delete();
                }
            } else {
                $alreadyAddedQty = 0;
                if ($item->getId()) {
                    $collection = $this->quoteCollection->create();
                    $bookingQuote = $this->getDataByField($item->getId(), 'item_id', $collection);
                    if ($bookingQuote) {
                        $alreadyAddedQty = $item->getQty();
                    }
                }
                if ($requestedQty > $availableQty || ($alreadyAddedQty != 0 && $alreadyAddedQty == $availableQty)) {
                    if (!empty($data['charged_per_count']) && $data['charged_per_count'] > 1) {
                        $availableQty = $availableQty / $data['charged_per_count'];
                        $availableQty = (int) $availableQty;
                    }
                    if ($alreadyAddedQty == $availableQty || $requestedQty > $availableQty) {
                        $isThrowError = true;
                    } else {
                        $item->setQty($availableQty)->save();
                    }
                    $error = 1;
                    $errorMessage = __(
                        'Only %1 quantity is available for %2 for slot %3.',
                        $availableQty,
                        $item->getName(),
                        $bookedSlotDate
                    );

                    // $cartData = [$item->getId() => ['qty' => (int) $availableQty]];
                    // if ($this->_getSession->hasQuote()) {
                    //     $this->_updateShoppingCart($cartData);
                    // }

                    if ($isThrowError) {
                        if ($this->_getSession->hasQuote()) {
                            $this->_getSession->getQuote()->collectTotals();
                            $this->_getSession->getQuote()->setHasError(true);
                            throw new \Magento\Framework\Exception\LocalizedException(
                                $errorMessage
                            );
                        } else {
                            return;
                        }
                    } else {
                        $this->messageManager->addError($errorMessage);
                        $this->_getSession->getQuote()->collectTotals();
                    }
                }
                // save slot item data in booking quote table
                if ($itemId = $item->getId()) {
                    $collection = $this->quoteCollection->create();
                    $bookingQuote = $this->getDataByField($itemId, 'item_id', $collection);
                    if (!empty($item->getQuoteId())) {
                        $quoteId = $item->getQuoteId();
                    } else {
                        $quoteId = $this->_getSession->getQuote()->getId();
                    }
                    if (!$bookingQuote) {
                        $data =  [
                            'item_id' => $itemId,
                            'slot_id' => $slotId,
                            'parent_slot_id' => $parentSlotId,
                            'slot_day_index' => $slotDayIndex,
                            'slot_date' => $selectedBookingDate,
                            'slot_time' => $selectedBookingTime,
                            'quote_id' => $quoteId,
                            'product_id' => $productId,
                            'qty' => $item->getQty()
                        ];
                        $this->quote->create()->setData($data)->save();
                    }
                }
            }
        }
    }

    /**
     * GetMaxGuestsAvailable
     *
     * @return array
     */
    public function getMaxGuestsAvailable()
    {
        $guestsArr = [];
        try {
            $items =  $this->_getSession->getQuote()->getAllVisibleItems();
            foreach ($items as $item) {
                $productAttrSetId = $item->getProduct()->getAttributeSetId();
                $tableAttrSetId = $this->getProductAttributeSetIdByLabel(
                    'Table Booking'
                );
                if ($this->isBookingProduct($item->getProductId())
                    && $productAttrSetId == $tableAttrSetId
                ) {
                    $data = $item->getBuyRequest()->getData();
                    $itemId = (int) $item->getId();
                    $paramData = $this->getParams();
                    if (!empty($paramData['item'][$itemId]['booking_date'])) {
                        $data = $paramData['item'][$itemId];
                    }
                    $selectedBookingDate = $data['booking_date'];
                    $selectedBookingTime = trim($data['booking_time']);
                    $bookedSlotDate = strtotime($selectedBookingDate . " " . $selectedBookingTime);
                    $noOfGuests = $item->getQty();
                    if (!empty($data['charged_per_count'])
                        && $data['charged_per_count'] > 1
                    ) {
                        $noOfGuests = $noOfGuests * $data['charged_per_count'];
                    }
                    if (!empty($guestsArr[$item->getProductId()])) {
                        if (!empty($guestsArr[$item->getProductId()][$bookedSlotDate])) {
                            $guestsArr[$item->getProductId()][$bookedSlotDate] += $noOfGuests;
                        } else {
                            $guestsArr[$item->getProductId()][$bookedSlotDate] = $noOfGuests;
                        }
                    } else {
                        $guestsArr[$item->getProductId()] = [
                            $bookedSlotDate => $noOfGuests
                        ];
                    }
                }
            }
        } catch (\Exception $e) {
            $this->logDataInLogger("Helper_Data_getMaxGuestsAvailable Exception : ".$e->getMessage());
        }
        return $guestsArr;
    }

    /**
     * Check is Guests Capacity available for selected Date range or not
     *
     * @param array $data
     * @param \Magento\Catalog\Model\Product $product
     * @param \Magento\Sales\Model\Order\Item $item
     */
    public function checkIsCapacityAvailableForDateRange($data, $product, $item)
    {
        $itemId = (int) $item->getId();
        $noOfGuestsRequested = $item->getProduct()->getCartQty();
        $paramData = $this->getParams();
        if (!empty($paramData['item'][$itemId]['qty'])) {
            $noOfGuestsRequested = $paramData['item'][$itemId]['qty'];
        }
        if (!empty($data['charged_per_count'])
            && $data['charged_per_count'] > 1
        ) {
            $noOfGuestsRequested = $noOfGuestsRequested * $data['charged_per_count'];
        }
        $error = false;
        $errorMessage = __("something went wrong !!!");
        $tableAttrSetId = $this->getProductAttributeSetIdByLabel(
            'Table Booking'
        );

        if (empty($data['booking_date']) || empty($data['booking_time'])) {
            return null;
        }
        
        // Check if capacity is available or not
        $selectedBookingDate = $data['booking_date'];
        $selectedBookingTime = $data['booking_time'];
        $bookedSlotDate = $selectedBookingDate." ".$selectedBookingTime;
        $slotDayIndex = $data['slot_day_index'];
        $formattedBookedSlotDate = date(
            "d M, Y",
            strtotime($selectedBookingDate)
        )." ".trim($selectedBookingTime);

        $maxCapacity = $product->getMaxCapacity();
        $info = $this->getBookingInfo($product->getId());

        if (!empty($info['is_booking']) && $info['is_booking'] && !empty($info['info'])) {
            $info  = $this->getJsonDecodedString(
                $info['info']
            );
        }
        if (!empty($info[$slotDayIndex][$data['parent_slot_id']]["slots_info"][$data['slot_id']])) {
            $maxCapacity = $info[$slotDayIndex][$data['parent_slot_id']]["slots_info"][$data['slot_id']]['qty'];
        }
        $availability = $maxCapacity;
        if ($product['price_charged_per_table'] == 2) {
            $availability = $availability / ($data['charged_per_count'] > 1 ? $data['charged_per_count'] : 1);
        }
        $errorMessage = __(
            'Only %1 Tables are left at %2 slot.',
            $availability,
            $formattedBookedSlotDate
        );
        if ($maxCapacity <= 0) {
            $error = true;
            $errorMessage = __(
                'All Tables are reserved at %1.',
                $formattedBookedSlotDate
            );
        }
        if ($error) {
            $this->_getSession->getQuote()->setHasError(true);
            throw new \Magento\Framework\Exception\LocalizedException(
                $errorMessage
            );
        }
        $items =  $this->_getSession->getQuote()->getAllVisibleItems();
        foreach ($items as $tempItem) {
            if (!$this->isBookingProduct($tempItem->getProductId())
                || $tempItem->getProduct()->getAttributeSetId() != $tableAttrSetId
            ) {
                continue;
            }

            $itemData = $tempItem->getBuyRequest()->getData();
            $paramData = $this->getParams();
            if (!empty($paramData['item'][$itemId]['booking_date'])) {
                $itemData = $paramData['item'][$itemId];
            }
            $tempSelectedBookingDate = $itemData['booking_date'];
            $tempSelectedBookingTime = $itemData['booking_time'];
            $tempBookedSlotDate = $tempSelectedBookingDate ." ". $tempSelectedBookingTime;
            $noOfGuests = $tempItem->getQty();
            if (!empty($itemData['charged_per_count'])
                && $itemData['charged_per_count'] > 1
            ) {
                $noOfGuests = $noOfGuests * $itemData['charged_per_count'];
            }
            if ($tempBookedSlotDate) {
                if (strtotime($bookedSlotDate) !== strtotime($tempBookedSlotDate)) {
                    $error = false;
                } elseif (($maxCapacity - $noOfGuests) >= $noOfGuestsRequested) {
                    $error = false;
                } else {
                    $error = true;
                    if ($maxCapacity - $noOfGuests <= 0) {
                        $errorMessage = __(
                            'All Tables are reserved at %1.',
                            $formattedBookedSlotDate
                        );
                    } else {
                        $availability = $maxCapacity - $noOfGuests;
                        if ($product['price_charged_per_table'] == 2) {
                            $availability = $availability / (
                                $itemData['charged_per_count'] > 1 ? $itemData['charged_per_count'] : 1
                            );
                        }
                        $errorMessage = __(
                            'Only %1 Tables are left at %2 slot.',
                            $availability,
                            $formattedBookedSlotDate
                        );
                    }
                }
            }
        }
        if ($error) {
            $this->_getSession->getQuote()->setHasError(true);
            throw new \Magento\Framework\Exception\LocalizedException(
                $errorMessage
            );
        }
    }

    /**
     * Process Table Booking Save Data
     *
     * @param array $data
     * @param \Magento\Catalog\Model\Product $product
     * @param \Magento\Sales\Model\Order\Item $item
     * @param \Webkul\AdvancedBookingSystem\Model\Quote $bookingQuote
     */
    public function checkItemQtyAvilableForTable($data, $product, $item, $bookingQuote)
    {
        $noOfGuestsRequested = $data['qty'];
        if (!empty($data['charged_per_count'])
            && $data['charged_per_count'] > 1
        ) {
            $noOfGuestsRequested = $noOfGuestsRequested * $data['charged_per_count'];
        }
        $error = false;
        $errorMessage = __("something went wrong !!!");
        $tableAttrSetId = $this->getProductAttributeSetIdByLabel(
            'Table Booking'
        );

        if (!empty($data['booking_date']) && !empty($data['booking_time'])) {
            // Check if capacity is available or not
            $selectedBookingDate = $data['booking_date'];
            $selectedBookingTime = $data['booking_time'];
            $bookedSlotDate = $selectedBookingDate." ".$selectedBookingTime;
            $slotDayIndex = $data['slot_day_index'];

            $formattedBookedSlotDate = date(
                "d M, Y",
                strtotime($selectedBookingDate)
            )." ".trim($selectedBookingTime);

            $maxCapacity = $product->getMaxCapacity();
            $info = $this->getBookingInfo($product->getId());

            if (!empty($info['is_booking']) && $info['is_booking'] && !empty($info['info'])) {
                $info  = $this->getJsonDecodedString(
                    $info['info']
                );
            }
            if (!empty($info[$slotDayIndex][$data['parent_slot_id']]["slots_info"][$data['slot_id']])) {
                $maxCapacity = $info[$slotDayIndex][$data['parent_slot_id']]["slots_info"][$data['slot_id']]['qty'];
            }
            $availability = $maxCapacity;
            if ($product['price_charged_per_table'] == 2) {
                $availability = $availability / ($data['charged_per_count'] > 1 ? $data['charged_per_count'] : 1);
            }
            $errorMessage = __(
                'Only %1 Tables are left at %2 slot.',
                $availability,
                $formattedBookedSlotDate
            );
            if ($maxCapacity <= 0) {
                $error = true;
                $errorMessage = __(
                    'All Tables are reserved at %1.',
                    $formattedBookedSlotDate
                );
            }

            if (!$error) {
                $noOfGuests = $bookingQuote->getQty();
                $itemData = $item->getBuyRequest()->getData();
                $itemId = (int) $item->getId();
                $paramData = $this->getParams();
                if (!empty($paramData['item'][$itemId])) {
                    $itemData = $paramData['item'][$itemId];
                }
                if (!empty($itemData['charged_per_count'])
                    && $itemData['charged_per_count'] > 1
                ) {
                    $noOfGuests = $noOfGuests * $itemData['charged_per_count'];
                }
                $noOfGuestsTotal = $noOfGuests + $noOfGuestsRequested;
                if ($maxCapacity < $noOfGuestsTotal) {
                    $error = true;
                    if ($maxCapacity - $noOfGuests <= 0) {
                        $errorMessage = __(
                            'All Tables are reserved at %1.',
                            $formattedBookedSlotDate
                        );
                    } else {
                        $availability = $maxCapacity - $noOfGuests;
                        if ($product['price_charged_per_table'] == 2) {
                            $availability = $availability / (
                                $itemData['charged_per_count'] > 1 ? $itemData['charged_per_count'] : 1
                            );
                        }
                        $errorMessage = __(
                            'Only %1 Tables are left at %2 slot.',
                            $availability,
                            $formattedBookedSlotDate
                        );
                    }
                }
            }
        }

        if ($error) {
            $this->_getSession->getQuote()->setHasError(true);
            throw new \Magento\Framework\Exception\LocalizedException(
                $errorMessage
            );
        }
    }
}
