<?php
/**
 * Webkul Software.
 *
 * @category  Webkul
 * @package   Webkul_AdvancedBookingSystem
 * @author    Webkul Software Private Limited
 * @copyright Webkul Software Private Limited (https://webkul.com)
 * @license   https://store.webkul.com/license.html
 */
namespace Webkul\AdvancedBookingSystem\Helper;

use Magento\Customer\Model\Session;
use Magento\Framework\Exception\MailException;

/**
 * Webkul AdvancedBookingSystem Helper Email.
 */
class Email extends \Magento\Framework\App\Helper\AbstractHelper
{
    public const XML_PATH_EMAIL_CONTACT = 'advancedbookingsystem/email/contact_template';
    public const XML_PATH_EMAIL_QUESTION = 'advancedbookingsystem/email/question_template';
    public const XML_PATH_EMAIL_CANCELLATION = 'advancedbookingsystem/email/cancellation_template';

    /**
     * @var \Magento\Framework\Translate\Inline\StateInterface
     */
    protected $inlineTranslation;

    /**
     * @var \Magento\Framework\Mail\Template\TransportBuilder
     */
    protected $transportBuilder;

    /**
     * Selected template id
     *
     * @var $template
     */
    protected $template;

    /**
     * @var \Magento\Store\Model\StoreManagerInterface
     */
    protected $storeManager;

    /**
     * @var \Magento\Framework\Message\ManagerInterface
     */
    protected $messageManager;

    /**
     * @param \Magento\Framework\App\Helper\Context $context
     * @param \Magento\Framework\Translate\Inline\StateInterface $inlineTranslation
     * @param \Magento\Framework\Mail\Template\TransportBuilder $transportBuilder
     * @param \Magento\Framework\Message\ManagerInterface $messageManager
     * @param \Magento\Store\Model\StoreManagerInterface $storeManager
     */
    public function __construct(
        \Magento\Framework\App\Helper\Context $context,
        \Magento\Framework\Translate\Inline\StateInterface $inlineTranslation,
        \Magento\Framework\Mail\Template\TransportBuilder $transportBuilder,
        \Magento\Framework\Message\ManagerInterface $messageManager,
        \Magento\Store\Model\StoreManagerInterface $storeManager
    ) {
        parent::__construct($context);
        $this->inlineTranslation = $inlineTranslation;
        $this->transportBuilder = $transportBuilder;
        $this->storeManager = $storeManager;
        $this->messageManager = $messageManager;
    }

    /**
     * Return store configuration value.
     *
     * @param string $path
     * @param int    $storeId
     *
     * @return mixed
     */
    protected function getConfigValue($path, $storeId)
    {
        return $this->scopeConfig->getValue(
            $path,
            \Magento\Store\Model\ScopeInterface::SCOPE_STORE,
            $storeId
        );
    }

    /**
     * Return store.
     *
     * @return Store
     */
    public function getStore()
    {
        return $this->storeManager->getStore();
    }

    /**
     * Return template id.
     *
     * @param string $xmlPath
     * @return mixed
     */
    public function getTemplateId($xmlPath)
    {
        return $this->getConfigValue($xmlPath, $this->getStore()->getStoreId());
    }

    /**
     * [generateTemplate description].
     *
     * @param Mixed $emailTemplateVariables
     * @param Mixed $senderInfo
     */
    public function generateTemplate($emailTemplateVariables, $senderInfo)
    {
        $senderEmail = $senderInfo['email'];
        $adminEmail = $this->getConfigValue(
            'trans_email/ident_general/email',
            $this->getStore()->getStoreId()
        );
        $receiverInfo['name'] = $this->getConfigValue(
            'trans_email/ident_general/name',
            $this->getStore()->getStoreId()
        );
        $receiverInfo['email'] = $adminEmail;
        $senderInfo['email'] = $senderEmail; //$adminEmail;
        $template = $this->transportBuilder->setTemplateIdentifier($this->template)
            ->setTemplateOptions(
                [
                        'area' => \Magento\Framework\App\Area::AREA_FRONTEND,
                        'store' => $this->storeManager->getStore()->getId(),
                    ]
            )
            ->setTemplateVars($emailTemplateVariables)
            ->setFrom($senderInfo)
            ->addTo($receiverInfo['email'], $receiverInfo['name']);
        $this->transportBuilder->setReplyTo($senderEmail);
        return $this;
    }

    /*transaction email template*/
    /**
     * [sendContactMailToAdmin description].
     *
     * @param Mixed $data
     * @param Mixed $emailTemplateVariables
     * @param Mixed $senderInfo
     */
    public function sendContactMailToAdmin($data, $emailTemplateVariables, $senderInfo)
    {
        $this->template = $this->getTemplateId(self::XML_PATH_EMAIL_CONTACT);

        $this->inlineTranslation->suspend();

        $this->generateTemplate($emailTemplateVariables, $senderInfo);
        try {
            $transport = $this->transportBuilder->getTransport();
            $transport->sendMessage();
        } catch (\Exception $e) {
            $this->messageManager->addError($e->getMessage());
        }

        $this->inlineTranslation->resume();
    }

    /**
     * [sendQuestionMailToAdmin description].
     *
     * @param Mixed $data
     * @param Mixed $emailTemplateVariables
     */
    public function sendQuestionMailToAdmin($data, $emailTemplateVariables)
    {
        $senderInfo = [
            'name' => $data['nick_name'],
            'email' => $this->getConfigValue(
                'trans_email/ident_support/email',
                $this->getStore()->getStoreId()
            )
        ];
        $this->template = $this->getTemplateId(self::XML_PATH_EMAIL_QUESTION);

        $this->inlineTranslation->suspend();

        $this->generateTemplate($emailTemplateVariables, $senderInfo);
        try {
            $transport = $this->transportBuilder->getTransport();
            $transport->sendMessage();
        } catch (\Exception $e) {
            $this->messageManager->addError($e->getMessage());
        }

        $this->inlineTranslation->resume();
    }

    /**
     * [sendCancellationMailToAdmin description].
     *
     * @param Mixed $emailTemplateVariables
     */
    public function sendCancellationMailToAdmin($emailTemplateVariables)
    {
        $senderInfo = [
            'name' => (string)__('Customer'),
            'email' => $this->getConfigValue(
                'trans_email/ident_support/email',
                $this->getStore()->getStoreId()
            )
        ];
        $this->template = $this->getTemplateId(self::XML_PATH_EMAIL_CANCELLATION);
        $this->inlineTranslation->suspend();

        $this->generateTemplate($emailTemplateVariables, $senderInfo);
        try {
            $transport = $this->transportBuilder->getTransport();
            $transport->sendMessage();
        } catch (\Exception $e) {
            $this->messageManager->addError($e->getMessage());
        }

        $this->inlineTranslation->resume();
    }
}
