<?php
/**
 * Webkul Software.
 *
 * @category  Webkul
 * @package   Webkul_AdvancedBookingSystem
 * @author    Webkul Software Private Limited
 * @copyright Webkul Software Private Limited (https://webkul.com)
 * @license   https://store.webkul.com/license.html
 */
namespace Webkul\AdvancedBookingSystem\Helper;

class Ics
{
    public const DATE_FORMAT = 'Ymd\THis';

    /**
     * @var array
     */
    protected $properties = [];

    /**
     * @var array
     */
    private $availableProperties = [
        'tzid',
        'description',
        'dtend',
        'dtstart',
        'location',
        'summary',
        'url',
        'organizer'
    ];

    /**
     * @param array $props
     */
    public function __construct($props = [])
    {
        $this->set($props);
    }

    /**
     * Set
     *
     * @param string $key
     * @param boolean $val
     * @return void
     */
    public function set($key, $val = false)
    {
        if (is_array($key)) {
            foreach ($key as $k => $v) {
                $this->set($k, $v);
            }
        } else {
            if (in_array($key, $this->availableProperties)) {
                $this->properties[$key] = $this->sanitizeValue($val, $key);
            }
        }
    }

    /**
     * GetContent
     *
     * @return string
     */
    public function getContent()
    {
        $rows = $this->buildProperties();
        return implode("\r\n", $rows);
    }

    /**
     * BuildProperties
     *
     * @return array
     */
    private function buildProperties()
    {
        // Build ICS properties - add header
        $icsProperties = [
            'BEGIN:VCALENDAR',
            'VERSION:2.0',
            'PRODID:-//hacksw/handcal//NONSGML v1.0//EN',
            'CALSCALE:GREGORIAN',
            'BEGIN:VEVENT'
        ];

        // Build ICS properties - add header
        $props = [];
        foreach ($this->properties as $key => $val) {
            if ($key === 'url') {
                $props[strtoupper($key . ';VALUE=URI')] = $val;
            } elseif ($key === 'dtstart') {
                $props[strtoupper($key . ';TZID=').$this->properties['tzid']] = $val;
            } elseif ($key === 'dtend') {
                $props[strtoupper($key . ';TZID=').$this->properties['tzid']] = $val;
            } else {
                $props[strtoupper($key)] = $val;
            }
        }

        // Set some default values
        $props['DTSTAMP'] = $this->formatTimeStamp('now');
        $props['UID'] = uniqid();

        // Append properties
        foreach ($props as $key => $val) {
            $icsProperties[] = "$key:$val";
        }

        // Build ICS properties - add footer
        $icsProperties[] = 'END:VEVENT';
        $icsProperties[] = 'END:VCALENDAR';

        return $icsProperties;
    }

    /**
     * SanitizeValue
     *
     * @param string $val
     * @param boolean $key
     * @return string
     */
    private function sanitizeValue($val, $key = false)
    {
        switch ($key) {
            case 'dtend':
            case 'dtstamp':
            case 'dtstart':
                $val = $this->formatTimeStamp($val);
                break;
            default:
                $val = $this->escapeString($val);
        }
        return $val;
    }

    /**
     * FormatTimeStamp
     *
     * @param int $timeStamp
     * @return string
     */
    private function formatTimeStamp($timeStamp)
    {
        $dt = new \DateTime($timeStamp);
        return $dt->format(self::DATE_FORMAT);
    }

    /**
     * EscapeString
     *
     * @param string $str
     * @return string
     */
    private function escapeString($str)
    {
        return preg_replace('/([\,;])/', '\\\$1', $str);
    }
}
