<?php

declare(strict_types=1);

/**
 * Webkul Software.
 *
 * @category  Webkul
 * @package   Webkul_AdvancedBookingSystem
 * @author    Webkul Software Private Limited
 * @copyright Webkul Software Private Limited (https://webkul.com)
 * @license   https://store.webkul.com/license.html
 */

namespace Webkul\AdvancedBookingSystem\Model\Resolver;

use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlAuthorizationException;
use Magento\Framework\GraphQl\Exception\GraphQlNoSuchEntityException;
use Magento\Framework\GraphQl\Exception\GraphQlInputException;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Query\Resolver\ContextInterface;
use Webkul\AdvancedBookingSystem\Helper\Data;
use Magento\QuoteGraphQl\Model\Cart\GetCartForUser;
use Magento\Framework\GraphQl\Query\Uid;
use Webkul\AdvancedBookingSystem\Model\ResourceModel\Quote\CollectionFactory as QuoteCollection;

/**
 * Resolver for AddItemToBookingQuote mutation
 */
class AddItemToBookingQuote implements ResolverInterface
{
    /**
     * @var Data
     */
    private $helper;
    
    /**
     * @var GetCartForUser
     */
    private $getCartForUser;
    
    /**
     * @var Uid
     */
    private $uidEncoder;
    
    /**
     * @var QuoteCollection
     */
    private $quoteCollection;

    /**
     *
     * @param Data $helper
     * @param GetCartForUser $getCartForUser
     * @param Uid $uidEncoder
     * @param QuoteCollection $quoteCollection
     */
    public function __construct(
        Data $helper,
        GetCartForUser $getCartForUser,
        Uid $uidEncoder,
        QuoteCollection $quoteCollection
    ) {
        $this->helper = $helper;
        $this->getCartForUser = $getCartForUser;
        $this->uidEncoder = $uidEncoder;
        $this->quoteCollection = $quoteCollection;
    }

    /**
     * AddItemToBookingQuote Resolver
     *
     * @param \Magento\Framework\GraphQl\Config\Element\Field $field
     * @param ContextInterface $context
     * @param ResolveInfo $info
     * @param array|null $value
     * @param array|null $args
     * @throws GraphQlException
     * @return mixed|Value
     */
    public function resolve(
        Field $field,
        $context,
        ResolveInfo $info,
        array $value = null,
        array $args = null
    ) {
        try {
            $item = null;
            if (empty($args['cartId'])) {
                throw new GraphQlInputException(__('Required parameter "cartId" is missing'));
            }
            if (empty($args['cartItemUid'])) {
                throw new GraphQlInputException(__('Required parameter "cartItemUid" is missing'));
            }

            $helper = $this->helper;

            $itemId = $this->uidEncoder->decode($args['cartItemUid']);
            $userId = (int)$context->getUserId();
            $storeId = (int)$context->getExtensionAttributes()->getStore()->getId();
            $cart = $this->getCartForUser->execute($args['cartId'], $userId, $storeId);
            $item = $cart->getItemById($itemId);

            if (!$item || !$item->getBuyRequest()) {
                throw new GraphQlNoSuchEntityException(__('Item doesn\'t have buy request'));
            }
            $data = $item->getBuyRequest()->getData();
            $productId = $item->getProduct()->getId();
            $itemId = (int) $item->getId();
            $product = $helper->getProduct($productId);
            $productSetId = $product->getAttributeSetId();
            $allowedAttrSetIDs = $helper->getAllowedAttrSetIDs();

            $eventAttrSetId = $helper->getProductAttributeSetIdByLabel(
                'Event Booking'
            );
            $tableAttrSetId = $helper->getProductAttributeSetIdByLabel(
                'Table Booking'
            );
            $appointmentAttrSetId = $helper->getProductAttributeSetIdByLabel(
                'Appointment Booking'
            );
            $rentalAttrSetId = $helper->getProductAttributeSetIdByLabel(
                'Rental Booking'
            );
            $hotelAttrSetId = $helper->getProductAttributeSetIdByLabel(
                'Hotel Booking'
            );

            if ($helper->isBookingProduct($productId)) {
                $isThrowError = 1;
                if (in_array($productSetId, $allowedAttrSetIDs) || $productSetId == $hotelAttrSetId) {
                    if ($appointmentAttrSetId == $productSetId) {
                        if (empty($args['bookingDate'])) {
                            throw new GraphQlInputException(__('Required parameter "bookingDate" is missing'));
                        }
                        if (empty($args['bookingTime'])) {
                            throw new GraphQlInputException(__('Required parameter "bookingTime" is missing'));
                        }
                        $data['charged_per_count'] = 1;
                        $data['booking_date'] = $args['bookingDate'];
                        $data['booking_time'] = $args['bookingTime'];
                        $data['slot_id'] = isset($args['slotId']) ? $args['slotId'] : 0;
                        $data['slot_day_index'] = isset($args['slotDayIndex']) ? $args['slotDayIndex'] : 0;
                        $data['parent_slot_id'] = isset($args['parentSlotId']) ? $args['parentSlotId'] : 0;
                        $helper->processBookingSave(
                            $data,
                            $product,
                            $item,
                            $isThrowError
                        );
                    } elseif ($eventAttrSetId == $productSetId) {
                        $helper->processEventBookingSave(
                            $data,
                            $product,
                            $item,
                            $isThrowError
                        );
                    } elseif ($productSetId == $hotelAttrSetId) {
                        if (empty($args['hotelQty'])) {
                            throw new GraphQlInputException(__('Required parameter "hotelQty" is missing'));
                        }
                        $data['hotel_qty'] = $args['hotelQty'];
                        $childId = $helper->getChildFromProductAttribute($productId, $data['super_attribute']);
                        $data['selected_configurable_option'] = $childId;

                        $helper->processHotelBookingSave(
                            $data,
                            $product,
                            $item,
                            $isThrowError
                        );
                    } elseif ($rentalAttrSetId == $productSetId) {
                        if (empty($args['bookingDateFrom'])) {
                            throw new GraphQlInputException(__('Required parameter "bookingDateFrom" is missing'));
                        }
                        $data['charged_per_count'] = 1;
                        $data['booking_date_from'] = $args['bookingDateFrom'];
                        $data['booking_date_to'] = isset($args['bookingDateTo']) ?
                        $args['bookingDateTo'] : $args['bookingDateFrom'];
                        $data['booking_from_time'] = isset($args['bookingFromTimeSlotIndex']) ?
                        $args['bookingFromTimeSlotIndex'] : 0;
                        $data['booking_to_time'] = isset($args['bookingToTimeSlotIndex']) ?
                        $args['bookingToTimeSlotIndex'] : 0;
                        $data['slot_id'] = isset($args['slotId']) ? $args['slotId'] : 0;
                        $data['slot_day_index'] = isset($args['slotDayIndex']) ? $args['slotDayIndex'] : 0;
                        $data['parent_slot_id'] = isset($args['parentSlotId']) ? $args['parentSlotId'] : 0;
                        $helper->processRentBookingSave(
                            $data,
                            $product,
                            $item,
                            $isThrowError
                        );
                    } elseif ($tableAttrSetId == $productSetId && $item->getId()) {
                        if (empty($args['bookingDate'])) {
                            throw new GraphQlInputException(__('Required parameter "bookingDate" is missing'));
                        }
                        if (empty($args['bookingTime'])) {
                            throw new GraphQlInputException(__('Required parameter "bookingTime" is missing'));
                        }
                        if (!isset($args['slotId'])) {
                            throw new GraphQlInputException(__('Required parameter "slotId" is missing'));
                        }
                        if (empty($args['slotDayIndex'])) {
                            throw new GraphQlInputException(__('Required parameter "slotDayIndex" is missing'));
                        }
                        if (!isset($args['parentSlotId'])) {
                            throw new GraphQlInputException(__('Required parameter "parentSlotId" is missing'));
                        }
                        $data['charged_per_count'] = 1;
                        $data['booking_date'] = $args['bookingDate'];
                        $data['booking_time'] = $args['bookingTime'];
                        $data['slot_id'] = $args['slotId'];
                        $data['slot_day_index'] = $args['slotDayIndex'];
                        $data['parent_slot_id'] = $args['parentSlotId'];
                        $itemId = (int) $item->getId();
                        $collection = $this->quoteCollection->create();
                        $bookingQuote = $helper->getDataByField($itemId, 'item_id', $collection);
                        if (!$bookingQuote) {
                            $helper->processTableBookingSave(
                                $data,
                                $product,
                                $item,
                                $isThrowError
                            );
                        }
                    }
                } elseif (isset($args['slotId']) && !empty($args['bookedQty'])) {
                    $data['slot_id'] = $args['slotId'];
                    $data['booked_qty'] = $args['bookedQty'];
                    $helper->processDefaultSlotData(
                        $data,
                        $item,
                        $productId,
                        $itemId,
                        $isThrowError
                    );
                }
            } else {
                throw new GraphQlInputException(__('Not a Booking product'));
            }
            if ($item->getHasError()) {
                $errorMsg = $item->getMessage();
                if ($item->getId()) {
                    $item->delete();
                }
                //for removing new line chars from errorMsg
                $errorMsg = str_replace("\n", '', $errorMsg);
                throw new GraphQlNoSuchEntityException(__($errorMsg . " item deleted from quote."));
            }
            return 'Item successfully added to Booking Quote.';
        } catch (LocalizedException $e) {
            if ($item && $item->getId()) {
                $item->delete();
            }
            throw new GraphQlNoSuchEntityException(__($e->getMessage() . " item deleted from quote."));
        }
    }
}
