<?php

declare(strict_types=1);

/**
 * Webkul Software.
 *
 * @category  Webkul
 * @package   Webkul_AdvancedBookingSystem
 * @author    Webkul Software Private Limited
 * @copyright Webkul Software Private Limited (https://webkul.com)
 * @license   https://store.webkul.com/license.html
 */

namespace Webkul\AdvancedBookingSystem\Model\Resolver;

use Magento\Authorization\Model\UserContextInterface;
use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlAuthorizationException;
use Magento\Framework\GraphQl\Exception\GraphQlNoSuchEntityException;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Webkul\AdvancedBookingSystem\Helper\Data;

/**
 * Resolver for AppointmentBookingInfo query
 */
class AppointmentBookingInfo implements ResolverInterface
{
    /**
     * @var Data
     */
    private $dataHelper;

    /**
     *
     * @param Data $dataHelper
     */
    public function __construct(
        Data $dataHelper
    ) {
        $this->dataHelper = $dataHelper;
    }

    /**
     * AppointmentBookingInfo Resolver
     *
     * @param \Magento\Framework\GraphQl\Config\Element\Field $field
     * @param ContextInterface $context
     * @param ResolveInfo $info
     * @param array|null $value
     * @param array|null $args
     * @throws GraphQlException
     * @return array
     */
    public function resolve(
        Field $field,
        $context,
        ResolveInfo $info,
        array $value = null,
        array $args = null
    ) {
        try {
            $productId = $args['productId'];
            $product = $this->dataHelper->getProduct($productId);
            if (!$product) {
                throw new GraphQlNoSuchEntityException(__('Product doesn\'t exist'));
            }
            $helper = $this->dataHelper;
            $productType = $product->getTypeId();
            $productSetId = $product->getAttributeSetId();
            $appointmentAttrSetId = $helper->getProductAttributeSetIdByLabel(
                'Appointment Booking'
            );
            if ($productType != 'booking' || $productSetId != $appointmentAttrSetId) {
                throw new GraphQlNoSuchEntityException(__('Not an Appointment Type Product'));
            }
            $BookingDates = $this->dataHelper->getValidBookingDates($product);
            $bookingSlotData = $this->dataHelper->getBookingInfo($productId);
            $bookingSlotData = $this->dataHelper->getJsonDecodedString($bookingSlotData['info']);
            $slotDuration = $product->getSlotDuration();
            $location = $product->getLocation();
            $today = date("l");
            $currentDate = strtotime($this->dataHelper->getCurrentDate());
            $todayIndex = $this->dataHelper->getDayIndexId($today);
            $emptySlot = [
                'from' => null,
                'to' => null,
                'qty' => null
            ];
            $todaySlots = [$emptySlot];
            $todaySlotsCount = 0;
            $isTodayHoliday =  $helper->isTodayHoliday($productId);
            if (isset($bookingSlotData[$todayIndex]) && !$isTodayHoliday) {
                $todaySlots = $bookingSlotData[$todayIndex];
                $todaySlotsCount = count($todaySlots);
            }
            $closed = ($currentDate >= strtotime($BookingDates['booking_available_from'])) &&
            $todaySlotsCount ? false : true;
            $bookingInfo = [
                'available' => !$closed,
                'slotDuration' => $slotDuration,
                'booking_available_from' => $BookingDates['booking_available_from'],
                'booking_available_to' => $BookingDates['booking_available_to'],
                'location' => $location,
                'todaySlots' => $todaySlots,
                'availabilityLabel' => $product['slot_for_all_days']==1 ? 'Everyday' : 'Today'
            ];
            return $bookingInfo;
        } catch (LocalizedException $e) {
            throw new GraphQlNoSuchEntityException(__($e->getMessage()));
        }
    }
}
