<?php

declare(strict_types=1);

/**
 * Webkul Software.
 *
 * @category  Webkul
 * @package   Webkul_AdvancedBookingSystem
 * @author    Webkul Software Private Limited
 * @copyright Webkul Software Private Limited (https://webkul.com)
 * @license   https://store.webkul.com/license.html
 */

namespace Webkul\AdvancedBookingSystem\Model\Resolver;

use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlAuthorizationException;
use Magento\Framework\GraphQl\Exception\GraphQlNoSuchEntityException;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Webkul\AdvancedBookingSystem\Helper\Data;

/**
 * Resolver for EventBookingInfo query
 */
class EventBookingInfo implements ResolverInterface
{
    /**
     * @var Data
     */
    private $dataHelper;

    /**
     *
     * @param Data $dataHelper
     */
    public function __construct(
        Data $dataHelper
    ) {
        $this->dataHelper = $dataHelper;
    }

    /**
     * EventBookingInfo Resolver
     *
     * @param \Magento\Framework\GraphQl\Config\Element\Field $field
     * @param ContextInterface $context
     * @param ResolveInfo $info
     * @param array|null $value
     * @param array|null $args
     * @throws GraphQlException
     * @return array
     */
    public function resolve(
        Field $field,
        $context,
        ResolveInfo $info,
        array $value = null,
        array $args = null
    ) {
        try {
            $productId = $args['productId'];
            $helper = $this->dataHelper;
            $EventBookingInfo = [];
            $product = $helper->getProduct($productId);
            if (!$product) {
                throw new GraphQlNoSuchEntityException(__('Product doesn\'t exist'));
            }
            $productType = $product->getTypeId();
            $productSetId = $product->getAttributeSetId();
            $eventAttrSetId = $helper->getProductAttributeSetIdByLabel(
                'Event Booking'
            );
            $emptyBookiOption = [
                'ticket_id' => null,
                'sku' => null,
                'title' => null,
                'description' => null,
                'price' => null,
                'is_in_stock' => null,
                'qty' => null,
            ];
            if ($productType != 'booking' || $productSetId != $eventAttrSetId) {
                throw new GraphQlNoSuchEntityException(__('Not an Event Type Product'));
            }

            $currentDate = $helper->getCurrentTime(true);
            $currentTime = $helper->getCurrentTime();
            $toDate = date('Y-m-d', strtotime($product['event_date_to']));
            $toTime = date('H:i', strtotime($product['event_date_to']));
            if (strtotime($toDate) > strtotime(date('Y-m-d', $currentDate))
                || (strtotime($toDate) == strtotime(date('Y-m-d', $currentDate)) && $currentTime <= strtotime($toTime))
            ) {
                $eventOptions = $helper->getEventOptions($product);
                if (!empty($eventOptions['event_ticket']['option_id'])) {
                    $productId = $product->getId();
                    $bookingInfo = $helper->getBookingInfo($productId);
                    $bookingSlotData = $helper->getJsonDecodedString($bookingInfo['info']);
                    $productFinalPrice = $product->getFinalPrice();
                    $EventBookingInfo['available'] = 1;
                    $EventBookingInfo['location'] = $product['location'];
                    $EventBookingInfo['product_price'] = $productFinalPrice;
                    $EventBookingInfo['show_map_loction'] = $product['show_map_loction'];
                    $EventBookingInfo['show_contact_button_to'] = ($product['show_contact_button_to'] == 0) ?
                    'Everyone' : 'Logged In Customer';
                    $dateFrom =  date(
                        'd F, Y h:i a - T',
                        strtotime($product['event_date_from'])
                    );
                    $dateTo =  date(
                        'd F, Y h:i a - T',
                        strtotime($product['event_date_to'])
                    );
                    $EventBookingInfo['event_date_from'] = $dateFrom;
                    $EventBookingInfo['event_date_to'] = $dateTo;
                    $EventBookingInfo['event_chart_available'] = $product['event_chart_available'];
                    if ($product['event_chart_available']) {
                        $chartPath = 'advancedbookingsystem/eventChartImage/' . $product['event_chart_image'];
                        $chartPath = $helper->getMediaUrl() . $chartPath;
                        $EventBookingInfo['event_chart_image'] = $chartPath;
                    } else {
                        $EventBookingInfo['event_chart_image'] = null;
                    }
                    $optionId = $eventOptions['event_ticket']['option_id'];
                    $optionValues = $eventOptions['event_ticket']['option_values'];
                    $EventBookingInfo['ticket_options'] = [];
                    if ($optionId && !empty($optionValues)) {
                        $this->processTicketOptions($product, $optionValues, $EventBookingInfo);
                    } else {
                        $EventBookingInfo['ticket_options'] = $emptyBookiOption;
                        $EventBookingInfo['available'] = 0;
                    }
                    $EventBookingInfo['priceRangeFrom'] = isset($priceRangeFrom) ?
                    $priceRangeFrom + $productFinalPrice : $productFinalPrice;
                    $EventBookingInfo['priceRangeTo'] = isset($priceRangeTo) ?
                    $priceRangeTo + $productFinalPrice : $productFinalPrice;
                }
            }
            if (empty($EventBookingInfo)) {
                $EventBookingInfo['available'] = 0;
                $EventBookingInfo['location'] = null;
                $EventBookingInfo['product_price'] = null;
                $EventBookingInfo['show_map_loction'] = null;
                $EventBookingInfo['show_contact_button_to'] = null;
                $EventBookingInfo['event_date_from'] = null;
                $EventBookingInfo['event_date_to'] = null;
                $EventBookingInfo['event_chart_available'] = null;
                $EventBookingInfo['event_chart_image'] = null;
                $EventBookingInfo['ticket_options'] = [$emptyBookiOption];
                $EventBookingInfo['priceRangeFrom'] = null;
                $EventBookingInfo['priceRangeTo'] = null;
            }
            return $EventBookingInfo;
        } catch (LocalizedException $e) {
            throw new GraphQlNoSuchEntityException(__($e->getMessage()));
        }
    }

    /**
     * Process ticket options
     *
     * @param object $product
     * @param array $optionValues
     * @param array $EventBookingInfo
     */
    private function processTicketOptions($product, $optionValues, &$EventBookingInfo)
    {
        $priceRangeFrom = 0;
        $priceRangeTo = 0;
        $priceIndex = 1;
        foreach ($optionValues as $key => $value) {
            $priceByStore = $this->dataHelper->currencyByStore(
                $value['price'],
                $product->getStore(),
                false
            );
            if ($priceIndex == 1) {
                $priceRangeFrom = $priceByStore;
                $priceRangeTo = $priceByStore;
            }
            if ($priceRangeFrom > $priceByStore) {
                $priceRangeFrom = $priceByStore;
            }
            if ($priceRangeTo < $priceByStore) {
                $priceRangeTo = $priceByStore;
            }
            $priceIndex++;
            $bookingOption['ticket_id'] = $value['option_type_id'];
            $bookingOption['sku'] = $value['sku'];
            $bookingOption['title'] = $value['title'];
            $bookingOption['description'] = $value['description'];
            $bookingOption['price'] = $value['price'];
            if ($value['is_in_stock'] && $value['qty']) {
                $bookingOption['is_in_stock'] = 1;
                $bookingOption['qty'] = $value['qty'];
            } else {
                $bookingOption['is_in_stock'] = 0;
                $bookingOption['qty'] = 0;
            }
            array_push($EventBookingInfo['ticket_options'], $bookingOption);
        }
    }
}
