<?php

declare(strict_types=1);

/**
 * Webkul Software.
 *
 * @category  Webkul
 * @package   Webkul_AdvancedBookingSystem
 * @author    Webkul Software Private Limited
 * @copyright Webkul Software Private Limited (https://webkul.com)
 * @license   https://store.webkul.com/license.html
 */

namespace Webkul\AdvancedBookingSystem\Model\Resolver;

use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlAuthorizationException;
use Magento\Framework\GraphQl\Exception\GraphQlNoSuchEntityException;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Webkul\AdvancedBookingSystem\Helper\Data;

/**
 * Resolver for HotelBookingInfo query
 */
class HotelBookingInfo implements ResolverInterface
{
    /**
     * @var Data
     */
    private $dataHelper;

    /**
     *
     * @param Data $dataHelper
     */
    public function __construct(
        Data $dataHelper
    ) {
        $this->dataHelper = $dataHelper;
    }

    /**
     * HotelBookingInfo Resolver
     *
     * @param \Magento\Framework\GraphQl\Config\Element\Field $field
     * @param ContextInterface $context
     * @param ResolveInfo $info
     * @param array|null $value
     * @param array|null $args
     * @throws GraphQlException
     * @return array
     */
    public function resolve(
        Field $field,
        $context,
        ResolveInfo $info,
        array $value = null,
        array $args = null
    ) {
        try {
            $productId = $args['productId'];
            $hotelBookingInfo = [];
            $helper = $this->dataHelper;
            $product = $helper->getProduct($productId);
            if (!$product) {
                throw new GraphQlNoSuchEntityException(__('Product doesn\'t exist'));
            }
            $productType = $product->getTypeId();
            $productSetId = $product->getAttributeSetId();
            $hotelAttrSetId = $helper->getProductAttributeSetIdByLabel(
                'Hotel Booking'
            );
            if ($productType != 'hotelbooking' || $productSetId != $hotelAttrSetId) {
                throw new GraphQlNoSuchEntityException(__('Not a Hotel Booking Type Product'));
            }
            $hotelBookingInfo['available'] = $product->isSaleable();
            $hotelBookingInfo['name'] = $product->getName();
            $hotelBookingInfo['sku'] = $product->getSku();
            $hotelBookingInfo['productPrice'] = $product->getFinalPrice();
            $hotelBookingInfo['allowedAdults'] = $product['max_adults'];
            $hotelBookingInfo['allowedKids'] = $product['max_kids'];
            $hotelBookingInfo['maxKidsAge'] = $product['max_age_kids'];
            $hotelBookingInfo['showMapLoction'] = $product['show_map_loction'];
            $hotelBookingInfo['askQuesEnable'] = $product['ask_a_ques_enable'];
            $hotelBookingInfo['shoNearByMap'] = $product['show_nearby_map'];
            $hotelBookingInfo['showContactButtonTo'] = ($product['show_contact_button_to'] == 0) ?
            'Everyone' : 'Logged In Customer';
            $priceChargedPer = "";
            if ($product['price_charged_per_hotel'] && $product['price_charged_per_hotel'] == 1) {
                $priceChargedPer = __("Night");
            }
            $hotelBookingInfo['priceChargedPer'] = $priceChargedPer;
            $stateName = $helper->getRegionName($product['hotel_state'], $product['hotel_country']);
            $formattedAddress = $product['hotel_address'] . ", " . $product['location']
                . $stateName . ", " . $product['hotel_country'];
            $hotelBookingInfo['location'] = $formattedAddress;
            $hotelBookingInfo['checkInTime'] = $product['check_in_time'];
            $hotelBookingInfo['checkOutTime'] = $product['check_out_time'];
            return $hotelBookingInfo;
        } catch (LocalizedException $e) {
            throw new GraphQlNoSuchEntityException(__($e->getMessage()));
        }
    }
}
