<?php

declare(strict_types=1);

/**
 * Webkul Software.
 *
 * @category  Webkul
 * @package   Webkul_AdvancedBookingSystem
 * @author    Webkul Software Private Limited
 * @copyright Webkul Software Private Limited (https://webkul.com)
 * @license   https://store.webkul.com/license.html
 */

namespace Webkul\AdvancedBookingSystem\Model\Resolver;

use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlAuthorizationException;
use Magento\Framework\GraphQl\Exception\GraphQlNoSuchEntityException;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Query\Uid;
use Webkul\AdvancedBookingSystem\Helper\Data;

/**
 * Resolver for OrderItemsCancellationInfo query
 */
class OrderItemsCancellationInfo implements ResolverInterface
{
    /**
     * @var Data
     */
    private $dataHelper;

    /**
     * @var Uid
     */
    private $uidEncoder;

    /**
     *
     * @param Data $dataHelper
     * @param Uid $uidEncoder
     */
    public function __construct(
        Data $dataHelper,
        Uid $uidEncoder
    ) {
        $this->dataHelper = $dataHelper;
        $this->uidEncoder = $uidEncoder;
    }

    /**
     * OrderItemsCancellationInfo Resolver
     *
     * @param \Magento\Framework\GraphQl\Config\Element\Field $field
     * @param ContextInterface $context
     * @param ResolveInfo $info
     * @param array|null $value
     * @param array|null $args
     * @throws GraphQlException
     * @return array
     */
    public function resolve(
        Field $field,
        $context,
        ResolveInfo $info,
        array $value = null,
        array $args = null
    ) {
        try {
            if (!isset($args['orderUid'])) {
                throw new GraphQlInputException(__('Order Uid is required.'));
            }
            $helper = $this->dataHelper;
            $orderId = $this->uidEncoder->decode($args['orderUid']);
            $order = $helper->getOrder($orderId);
            $orderItemsCancellationInfo = [];
            foreach ($order->getAllVisibleItems() as $item) {
                $productId = $item->getProductId();
                $isBookingProduct = $helper->isBookingProduct($productId);
                if (!$isBookingProduct) {
                    continue;
                }
                $itemId = $item->getId();
                $cancellationAvail = $helper->
                                getConfigData('advancedbookingsystem/cancellation/cancellation_available');
                $cancelStatus = $helper->getCancellationStatus($orderId, $itemId);
                $cancelInfo = $helper->getCancellationInfo($orderId, $itemId);
                $cancellationInfo = [
                    'itemId' => $itemId,
                    'cancellationAvail' => false,
                    'requested' => false,
                    'cancelled' => false,
                ];
                if ($cancellationAvail && $cancelStatus &&
                    $item->getIsCancellationAvailable()
                ) {
                    if ($item->canRefund() && empty($cancelInfo->getData())) {
                        $cancellationInfo['cancellationAvail'] = true;
                    } elseif (!$item->canRefund() && $cancelInfo->getStatus() == 1) {
                        $cancellationInfo['cancellationAvail'] = true;
                        $cancellationInfo['requested'] = true;
                        $cancellationInfo['cancelled'] = true;
                    } elseif ($item->canRefund()) {
                        $cancellationInfo['cancellationAvail'] = true;
                        $cancellationInfo['requested'] = true;
                    }
                }
                $orderItemsCancellationInfo[] = $cancellationInfo;
            }
            return $orderItemsCancellationInfo;
        } catch (LocalizedException $e) {
            throw new GraphQlNoSuchEntityException(__($e->getMessage()));
        }
    }
}
