<?php

declare(strict_types=1);

/**
 * Webkul Software.
 *
 * @category  Webkul
 * @package   Webkul_AdvancedBookingSystem
 * @author    Webkul Software Private Limited
 * @copyright Webkul Software Private Limited (https://webkul.com)
 * @license   https://store.webkul.com/license.html
 */

namespace Webkul\AdvancedBookingSystem\Model\Resolver;

use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlAuthorizationException;
use Magento\Framework\GraphQl\Exception\GraphQlNoSuchEntityException;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Webkul\AdvancedBookingSystem\Helper\Data;

/**
 * Resolver for RentBookingInfo query
 */
class RentBookingInfo implements ResolverInterface
{
    /**
     * @var Data
     */
    private $dataHelper;

    /**
     *
     * @param Data $dataHelper
     */
    public function __construct(
        Data $dataHelper,
    ) {
        $this->dataHelper = $dataHelper;
    }

    /**
     * RentBookingInfo Resolver
     *
     * @param \Magento\Framework\GraphQl\Config\Element\Field $field
     * @param ContextInterface $context
     * @param ResolveInfo $info
     * @param array|null $value
     * @param array|null $args
     * @throws GraphQlException
     * @return array
     */
    public function resolve(
        Field $field,
        $context,
        ResolveInfo $info,
        array $value = null,
        array $args = null
    ) {
        try {
            $productId = $args['productId'];
            $helper = $this->dataHelper;
            $RentBookingInfo = [];
            $product = $helper->getProduct($productId);
            if (!$product) {
                throw new GraphQlNoSuchEntityException(__('Product doesn\'t exist'));
            }
            $productType = $product->getTypeId();
            $productSetId = $product->getAttributeSetId();
            $rentAttrSetId = $helper->getProductAttributeSetIdByLabel(
                'Rental Booking'
            );
            if ($productType != 'booking' || $productSetId != $rentAttrSetId) {
                throw new GraphQlNoSuchEntityException(__('Not a Rental Type Product'));
            }
            $validBookingDates = $helper->getValidBookingDates($product);
            if (!is_array($validBookingDates)) {
                throw new GraphQlNoSuchEntityException(__('Not a valid Rent Type Product'));
            }
            $rentOptions = $helper->getRentOptions($product);
            if (empty($rentOptions['choose_rent_type']['option_id'])) {
                throw new GraphQlNoSuchEntityException(__('Not a valid Rent Type Product'));
            }
            $bookingAvailableFrom = $validBookingDates['booking_available_from'];
            $bookingAvailableTo = $validBookingDates['booking_available_to'];
            $RentBookingInfo['bookingAvailableFrom'] = $bookingAvailableFrom;
            $RentBookingInfo['bookingAvailableTo'] = $bookingAvailableTo;
            $RentBookingInfo['available'] = 1;
            $productFinalPrice = $product->getFinalPrice();
            $sku = $product->getSku();
            $name = $product->getName();
            $RentBookingInfo['name'] = $name;
            $RentBookingInfo['sku'] = $sku;
            $RentBookingInfo['productPrice'] = $productFinalPrice;
            $RentBookingInfo['location'] = $product['location'];
            $RentBookingInfo['showMapLoction'] = $product['show_map_loction'];
            $RentBookingInfo['showContactButtonTo'] = ($product['show_contact_button_to'] == 0) ?
            'Everyone' : 'Logged In Customer';
            $optionId = $rentOptions['choose_rent_type']['option_id'];
            $optionValues = $rentOptions['choose_rent_type']['option_values'];
            $isTodayHoliday = $helper->isTodayHoliday($productId);
            if ($optionId && !empty($optionValues)) {
                $priceChargedPer = '';
                $RentBookingInfo['rentTypeOptions'] = [];
                foreach ($optionValues as $key => $value) {
                    $priceByStore = $helper->currencyByStore(
                        $value['price'],
                        $product->getStore(),
                        false
                    );
                    $priceChargedPer = $priceByStore;
                    $bookingOption['rentTypeId'] = $value['option_type_id'];
                    if ($value['title'] == 'Hourly Basis') {
                        $bookingOption['title'] = 'Hourly Basis';
                        $bookingOption['chargePerTxt'] = __('Charged Per Hour');
                        $bookingOption['rentType'] = 'h';
                        $bookingOption['availableToday'] = !$isTodayHoliday;
                    }
                    if ($value['title'] == 'Daily Basis') {
                        $bookingOption['title'] = 'Daily Basis';
                        $bookingOption['chargePerTxt'] = __('Charged Per Day');
                        $bookingOption['rentType'] = 'd';
                        $bookingOption['availableToday'] = 1;
                    }
                    $bookingOption['price'] = $priceChargedPer;
                    array_push($RentBookingInfo['rentTypeOptions'], $bookingOption);
                }
            } else {
                $RentBookingInfo['rentTypeOptions'] = [];
                $RentBookingInfo['available'] = 0;
            }
            
            if (count($RentBookingInfo['rentTypeOptions'])==1
             && $RentBookingInfo['rentTypeOptions'][0]['rentType']=='h'
             && $isTodayHoliday
             ) {
                $RentBookingInfo['available'] = 0;
            }
            
            return $RentBookingInfo;
        } catch (LocalizedException $e) {
            throw new GraphQlNoSuchEntityException(__($e->getMessage()));
        }
    }
}
