<?php

declare(strict_types=1);

/**
 * Webkul Software.
 *
 * @category  Webkul
 * @package   Webkul_AdvancedBookingSystem
 * @author    Webkul Software Private Limited
 * @copyright Webkul Software Private Limited (https://webkul.com)
 * @license   https://store.webkul.com/license.html
 */

namespace Webkul\AdvancedBookingSystem\Model\Resolver;

use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlAuthorizationException;
use Magento\Framework\GraphQl\Exception\GraphQlNoSuchEntityException;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Query\Resolver\ContextInterface;
use Magento\Sales\Api\OrderRepositoryInterface;
use Webkul\AdvancedBookingSystem\Helper\Data;
use Webkul\AdvancedBookingSystem\Helper\Email;
use Webkul\AdvancedBookingSystem\Model\CancellationFactory;

/**
 * Resolver for RequestItemCancellation mutation
 */
class RequestItemCancellation implements ResolverInterface
{
    /**
     * @var Data
     */
    private $dataHelper;
    
    /**
     * @var OrderRepositoryInterface
     */
    private $orderRepository;
    
    /**
     * @var CancellationFactory
     */
    private $cancellationFactory;
    
    /**
     * @var Email
     */
    private $emailHelper;

    /**
     *
     * @param Data $dataHelper
     * @param OrderRepositoryInterface $orderRepository
     * @param CancellationFactory $cancellationFactory
     * @param Email $emailHelper
     */
    public function __construct(
        Data $dataHelper,
        OrderRepositoryInterface $orderRepository,
        CancellationFactory $cancellationFactory,
        Email $emailHelper
    ) {
        $this->dataHelper = $dataHelper;
        $this->orderRepository = $orderRepository;
        $this->cancellationFactory = $cancellationFactory;
        $this->emailHelper = $emailHelper;
    }

    /**
     * RequestItemCancellation Resolver
     *
     * @param \Magento\Framework\GraphQl\Config\Element\Field $field
     * @param ContextInterface $context
     * @param ResolveInfo $info
     * @param array|null $value
     * @param array|null $args
     * @throws GraphQlException
     * @return mixed|Value
     */
    public function resolve(
        Field $field,
        $context,
        ResolveInfo $info,
        array $value = null,
        array $args = null
    ) {
        try {
            if (!isset($args['orderItemId'])) {
                throw new GraphQlInputException(__('Item Uid is required.'));
            }
            $itemId = $args['orderItemId'];
            $helper = $this->dataHelper;
            try {
                $item = $helper->getOrderItem($itemId);
                $itemId = $item->getItemId();
                $orderId = $item->getOrderId();
                $order = $this->orderRepository->get($orderId);
                $status = $helper->getCancellationStatus($orderId, $item->getId());
                $invoiceItem = $helper->getInvoiceItemByOrderItemId($itemId);
                $cancellationAvail = $helper->
                getConfigData('advancedbookingsystem/cancellation/cancellation_available');
                $cancelInfo = $helper->getCancellationInfo($orderId, $itemId);
            } catch (\E $e) {
                $status = false;
            }

            if ($cancellationAvail && $status && $item->getIsCancellationAvailable() && empty($cancelInfo->getData())
            && $item->canRefund()) {
                $cancelCharge = $helper->getConfigData(
                    'advancedbookingsystem/cancellation/cancellation_charge'
                );
                $cancelAmount = 0;
                if ($cancelCharge) {
                    $cancelAmount = ($this->getTotalAmount($invoiceItem) * $cancelCharge) * .01;
                }

                $cancellation = $this->cancellationFactory->create();
                $cancellation->setOrderId($orderId);
                $cancellation->setOrderItemId($item->getId());
                $cancellation->setCreditMemoId('');
                $cancellation->setCancellationCharge($cancelAmount);
                $cancellation->setStatus(0);
                $cancellation->save();

                $this->sendCancellationEmail($orderId, $invoiceItem);
                $msg = "Cancellation Request Send";
                if ($cancelAmount>0) {
                    $msg = $msg.", ".$order->formatPriceTxt($cancelAmount).
                    " will be deducted from refund as cancellation charge.";
                }
                return $msg;
            } else {
                return "Cancellation is not allowed";
            }
        } catch (LocalizedException $e) {
            throw new GraphQlNoSuchEntityException(__($e->getMessage()));
        }
    }

    /**
     * Return the total amount minus discount
     *
     * @param OrderItem|InvoiceItem|CreditmemoItem $item
     * @return mixed
     */
    protected function getTotalAmount($item)
    {
        $totalAmount = $item->getRowTotal()
            + $item->getTaxAmount()
            + $item->getDiscountTaxCompensationAmount()
            + $item->getWeeeTaxAppliedRowAmount()
            - $item->getDiscountAmount();

        return $totalAmount;
    }

    /**
     * Cancellation email to admin
     *
     * @param int $orderId
     * @param object $item
     * @return null
     */
    protected function sendCancellationEmail($orderId, $item)
    {
        $data = [];
        $order = $this->orderRepository->get($orderId);

        $data['order_increment_id'] = $order->getIncrementId();
        $data['customer_name'] = (string)__('Admin');
        $data['product_name'] = $item->getName();
        $data['product_sku'] = $item->getSku();
        $data['subject'] = (string)__('Cancellation request generated');

        $this->emailHelper->sendCancellationMailToAdmin($data);
    }
}
