<?php

declare(strict_types=1);

/**
 * Webkul Software.
 *
 * @category  Webkul
 * @package   Webkul_AdvancedBookingSystem
 * @author    Webkul Software Private Limited
 * @copyright Webkul Software Private Limited (https://webkul.com)
 * @license   https://store.webkul.com/license.html
 */

namespace Webkul\AdvancedBookingSystem\Model\Resolver;

use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlAuthorizationException;
use Magento\Framework\GraphQl\Exception\GraphQlNoSuchEntityException;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\Stdlib\DateTime\TimezoneInterface;
use Webkul\AdvancedBookingSystem\Helper\Data;

/**
 * Resolver for SlotsInfo query
 */
class SlotsInfo implements ResolverInterface
{
    /**
     * @var Data
     */
    private $dataHelper;
    
    /**
     * @var TimezoneInterface
     */
    private $date;

    /**
     *
     * @param Data $dataHelper
     * @param TimezoneInterface $date
     */
    public function __construct(
        Data $dataHelper,
        TimezoneInterface $date
    ) {
        $this->dataHelper = $dataHelper;
        $this->date = $date;
    }

    /**
     * SlotsInfo Resolver
     *
     * @param \Magento\Framework\GraphQl\Config\Element\Field $field
     * @param ContextInterface $context
     * @param ResolveInfo $info
     * @param array|null $value
     * @param array|null $args
     * @throws GraphQlException
     * @return array
     */
    public function resolve(
        Field $field,
        $context,
        ResolveInfo $info,
        array $value = null,
        array $args = null
    ) {
        try {
            $productId = $args['productId'];
            $givenDate = $args['date'];
            $helper = $this->dataHelper;
            $product  = $helper->getProduct($productId);
            if (!$product) {
                throw new GraphQlNoSuchEntityException(__('Product doesn\'t exist'));
            }
            $isBookingProduct = $helper->isBookingProduct($productId);
            if (!$isBookingProduct) {
                throw new GraphQlNoSuchEntityException(__('Not a booking type product.'));
            }
            $currentDate = strtotime($givenDate);
            $currentLocalTime = $this->date->scopeTimeStamp();
            $validBookingDates = $helper->getValidBookingDates($product);
            $arr = [];
            $flag = false;
            if (!is_array($validBookingDates)) {
                throw new GraphQlNoSuchEntityException(__('Empty booking dates for this product.'));
            }
            
            $isTodayHoliday = $helper->isTodayHoliday($productId, date('Y-m-d', $currentDate));
            if ($isTodayHoliday) {
                return [];
            }

            $bookingAvailableFrom = $validBookingDates['booking_available_from'];
            $bookingAvailableTo = $validBookingDates['booking_available_to'];
            if (!empty($bookingAvailableFrom) && strtotime($bookingAvailableFrom) > $currentDate
                || !empty($bookingAvailableTo) && strtotime($bookingAvailableTo) < $currentDate
            ) {
                return [];
            }
            $fromDay = date("l", strtotime($bookingAvailableFrom));
            $fromDayIndex = $helper->getDayIndexId($fromDay);
            $bookingInfo = $helper->getBookingInfo($productId);
            $bookingSlotData = $helper->getJsonDecodedString($bookingInfo['info']);
            $unavailableDates = $helper->getUnvailableDates(
                $bookingSlotData,
                $bookingAvailableFrom,
                $bookingAvailableTo
            );
            $bookedData = $helper->getBookedAppointmentDates($productId);
            if (in_array($givenDate, $unavailableDates)) {
                return [];
            }
            $dateRange = [];
            if (!$product['available_every_week']) {
                $dateRange = $helper->calculateBookedDatesFromRange(
                    $bookingAvailableFrom,
                    $bookingAvailableTo,
                    "Y-m-d"
                );
                $range = $helper->calculateBookedDatesFromRange($bookingAvailableFrom, $bookingAvailableTo);
                if (count($range) != 1) {
                    $noOfDays = count($range) + 1;
                } else {
                    $noOfDays = ($bookingAvailableFrom == $bookingAvailableTo) ? 1 : count($range) + 1;
                }
                $flag = $this->processWeekArr($noOfDays, $fromDayIndex, $arr);
            }
            if (!empty($dateRange)) {
                $dateRange[] = date('Y-m-d', strtotime(end($dateRange) . ' +1 day'));
            }
            $bookingInfo = $helper->getBookingInfo($productId);
            $bookingSlotData = $helper->getJsonDecodedString($bookingInfo['info']);

            // Calculated current day slot data
            $today = date("l", $currentDate);
            $todayIndex = $helper->getDayIndexId($today);
            if (empty($bookingSlotData[1])) {
                $bookingSlotData[1] = [];
            }
            if (empty($bookingSlotData[2])) {
                $bookingSlotData[2] = [];
            }
            if (empty($bookingSlotData[3])) {
                $bookingSlotData[3] = [];
            }
            if (empty($bookingSlotData[4])) {
                $bookingSlotData[4] = [];
            }
            if (empty($bookingSlotData[5])) {
                $bookingSlotData[5] = [];
            }
            if (empty($bookingSlotData[6])) {
                $bookingSlotData[6] = [];
            }
            if (empty($bookingSlotData[7])) {
                $bookingSlotData[7] = [];
            }

            $availableSlotArr = [];
            $weekDateRange = $this->dataHelper->calculateBookedDatesFromRange(
                date("Y-m-d", strtotime('monday this week', $currentDate)),
                date("Y-m-d", strtotime('+1 day', strtotime("sunday this week", $currentDate))),
                "Y-m-d"
            );
            foreach ($weekDateRange as $key => $date) {
                $dayKey = $key + 1;
                $dayValue = $bookingSlotData[$dayKey];
                $availableSlotArr[$dayKey] = [];
                if (count($dayValue)
                    && (($flag && in_array($dayKey, $arr)) || !$flag)
                    && (($dayKey >= $todayIndex) && in_array($date, $dateRange) || !$flag)
                ) {
                    foreach ($dayValue as $key => $value) {
                        $availableSlotArr[$dayKey][$key] = $value['slots_info'];
                    }
                }
            }

            $availableTodaySlotArr = $availableSlotArr;
            $preventDuration = (float)$product['prevent_scheduling_before'];
            $currentTime = strtotime('+' . $preventDuration . ' minutes', $currentLocalTime);
            if ($currentDate === strtotime($bookingAvailableFrom)) {
                $this->processTodayAvailSlots(
                    $availableTodaySlotArr,
                    $fromDayIndex,
                    $currentTime
                );
            }
            $availableSlotSortArr = $this->processSortSlots($availableTodaySlotArr, $bookedData, $currentDate);
            if (empty($availableSlotSortArr) || !isset($availableSlotSortArr[$todayIndex])) {
                return [];
            }

            return $availableSlotSortArr[$todayIndex];
        } catch (LocalizedException $e) {
            throw new GraphQlNoSuchEntityException(__($e->getMessage()));
        }
    }

    /**
     * Sorting slots
     *
     * @param array $availableTodaySlotArr
     * @param array $bookedData
     * @param int $currentDate
     * @return array $availableSlotSortArr
     */
    private function processSortSlots($availableTodaySlotArr, $bookedData, $currentDate)
    {
        $availableSlotSortArr = [];
        $amTimeArray = [];
        $pmTimeArray = [];
        for ($i = 1; $i <= 7; $i++) {
            if (isset($availableTodaySlotArr[$i])) {
                foreach ($availableTodaySlotArr[$i] as $key => $value) {
                    foreach ($value as $slotKey => $slotValue) {
                        if ($this->checkSlotBooked($bookedData, $currentDate, $slotValue)) {
                            continue;
                        }
                        $timeArr = explode(' ', trim($slotValue['time']));
                        if ($timeArr[1] == 'am') {
                            $amTimeArray[$i][$slotKey]['slot_id'] = isset($slotValue['slot_id']) ?
                                $slotValue['slot_id'] : $slotKey;
                            $amTimeArray[$i][$slotKey]['time'] = $slotValue['time'];
                            $amTimeArray[$i][$slotKey]['qty'] = $slotValue['qty'];
                            $amTimeArray[$i][$slotKey]['parent_slot_id'] = $key;
                        } elseif ($timeArr[1] == 'pm') {
                            $amTimeArray[$i][$slotKey]['slot_id'] = isset($slotValue['slot_id']) ?
                                $slotValue['slot_id'] : $slotKey;
                            $pmTimeArray[$i][$slotKey]['time'] = $slotValue['time'];
                            $pmTimeArray[$i][$slotKey]['qty'] = $slotValue['qty'];
                            $pmTimeArray[$i][$slotKey]['parent_slot_id'] = $key;
                        }
                    }
                }
            }
        }
        $availableSlotSortArr = array_replace_recursive($amTimeArray, $pmTimeArray);
        return $availableSlotSortArr;
    }

    /**
     * Process Week array
     *
     * @param int $noOfDays
     * @param int $fromDayIndex
     * @param array $arr
     * @return bool $flag
     */
    private function processWeekArr($noOfDays, $fromDayIndex, &$arr)
    {
        $flag = false;
        if ($noOfDays < 7) {
            if ($fromDayIndex <= 7) {
                $arr[1] = $fromDayIndex;
                for ($i = 2; $i <= $noOfDays; $i++) {
                    if ($arr[$i - 1] + 1 <= 7) {
                        $arr[$i] = $arr[$i - 1] + 1;
                    } else {
                        $arr[$i] = ($arr[$i - 1] + 1) - 7;
                    }
                }
            }
            if (count($arr) > 0) {
                $flag = true;
            }
        }
        return $flag;
    }

    /**
     * Process today available slots
     *
     * @param array $availableTodaySlotArr
     * @param int $fromDayIndex
     * @param int $currentTime
     */
    private function processTodayAvailSlots(&$availableTodaySlotArr, $fromDayIndex, $currentTime)
    {
        foreach ($availableTodaySlotArr[$fromDayIndex] as $key => $value) {
            foreach ($value as $slotKey => $slotValue) {
                $availableTodaySlotArr[$fromDayIndex][$key][$slotKey]['timestring'] = strtotime($slotValue['time']);
                $availableTodaySlotArr[$fromDayIndex][$key][$slotKey]['slot_id'] = $slotKey;
                if ($currentTime >= strtotime($slotValue['time'])) {
                    unset($availableTodaySlotArr[$fromDayIndex][$key][$slotKey]);
                }
            }
        }
    }

    /**
     * Check if given slot is booked already
     *
     * @param array $bookedData
     * @param int $currentDate
     * @param array $slotValue
     */
    private function checkSlotBooked($bookedData, $currentDate, $slotValue)
    {
        if (!empty($bookedData[$currentDate][$slotValue['time']])) {
            $remainingQty = $slotValue['qty'] - $bookedData[$currentDate][$slotValue['time']];
            if ($remainingQty <= 0) {
                return true;
            }
        }
        return false;
    }
}
