<?php

declare(strict_types=1);

/**
 * Webkul Software.
 *
 * @category  Webkul
 * @package   Webkul_AdvancedBookingSystem
 * @author    Webkul Software Private Limited
 * @copyright Webkul Software Private Limited (https://webkul.com)
 * @license   https://store.webkul.com/license.html
 */

namespace Webkul\AdvancedBookingSystem\Model\Resolver;

use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlAuthorizationException;
use Magento\Framework\GraphQl\Exception\GraphQlNoSuchEntityException;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Query\Resolver\ContextInterface;
use Webkul\AdvancedBookingSystem\Helper\Data;

/**
 * Resolver for SubmitAnswer mutation
 */
class SubmitAnswer implements ResolverInterface
{
    /**
     * @var Data
     */
    private $dataHelper;
    
    /**
     * @var \Webkul\AdvancedBookingSystem\Model\AnswerFactory
     */
    private $answerModel;
    
    /**
     * @var \Magento\Framework\Stdlib\DateTime\DateTime
     */
    private $date;

    /**
     *
     * @param Data $dataHelper
     * @param \Webkul\AdvancedBookingSystem\Model\AnswerFactory $answerModel
     * @param \Magento\Framework\Stdlib\DateTime\DateTime $date
     */
    public function __construct(
        Data $dataHelper,
        \Webkul\AdvancedBookingSystem\Model\AnswerFactory $answerModel,
        \Magento\Framework\Stdlib\DateTime\DateTime $date
    ) {
        $this->dataHelper = $dataHelper;
        $this->answerModel = $answerModel;
        $this->date = $date;
    }

    /**
     * SubmitAnswer Resolver
     *
     * @param \Magento\Framework\GraphQl\Config\Element\Field $field
     * @param ContextInterface $context
     * @param ResolveInfo $info
     * @param array|null $value
     * @param array|null $args
     * @throws GraphQlException
     * @return mixed|Value
     */
    public function resolve(
        Field $field,
        $context,
        ResolveInfo $info,
        array $value = null,
        array $args = null
    ) {
        try {
            if (!isset($args['productId'])) {
                throw new GraphQlInputException(__('Product Id is required.'));
            }
            $productId = $args['productId'];
            $helper = $this->dataHelper;
            $userId = $context->getUserId();
            $product = $helper->getProduct($productId);
            if (!$product) {
                throw new GraphQlNoSuchEntityException(__('Product doesn\'t exist'));
            }
            $productType = $product->getTypeId();
            $productSetId = $product->getAttributeSetId();
            $hotelAttrSetId = $helper->getProductAttributeSetIdByLabel(
                'Hotel Booking'
            );
            if ($productType != 'hotelbooking' || $productSetId != $hotelAttrSetId) {
                throw new GraphQlNoSuchEntityException(__('Not a Hotel Booking Type Product'));
            }
            if (!$product['ask_a_ques_enable']) {
                throw new GraphQlInputException(__('Ask a question is not enabled for this product'));
            }
            if (!isset($args['customerNickname']) && $args['customerNickname'] == "") {
                throw new GraphQlInputException(__('Please enter your nick name'));
            } elseif (!isset($args['userAnswer']) && $args['userAnswer'] == "") {
                throw new GraphQlInputException(__('Please enter your answer'));
            } elseif (!isset($args['questionId']) && $args['questionId'] == "") {
                throw new GraphQlInputException(__('Question id is required'));
            }
            $data['nick_name'] = $args['customerNickname'];
            $data['answer'] = $args['userAnswer'];
            $data['question_id'] = $args['questionId'];
            $data['customer_id'] = $userId ? $userId : 0;
            $data['status'] = 1;
            $data['created_at'] = $this->date->gmtDate();
            $data['updated_at'] = $this->date->gmtDate();
            $model = $this->answerModel->create()->setData($data)->save();
            $id = $model->getId();
            if ($id) {
                return (__('Your answer has been submitted successfully'));
            } else {
                return (__('Something went wrong !!!'));
            }
        } catch (LocalizedException $e) {
            throw new GraphQlNoSuchEntityException(__($e->getMessage()));
        }
    }
}
