<?php

declare(strict_types=1);

/**
 * Webkul Software.
 *
 * @category  Webkul
 * @package   Webkul_AdvancedBookingSystem
 * @author    Webkul Software Private Limited
 * @copyright Webkul Software Private Limited (https://webkul.com)
 * @license   https://store.webkul.com/license.html
 */

namespace Webkul\AdvancedBookingSystem\Model\Resolver;

use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlAuthorizationException;
use Magento\Framework\GraphQl\Exception\GraphQlNoSuchEntityException;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Query\Resolver\ContextInterface;
use Webkul\AdvancedBookingSystem\Helper\Data;

/**
 * Resolver for SubmitQuestion mutation
 */
class SubmitQuestion implements ResolverInterface
{
    /**
     * @var Data
     */
    private $dataHelper;
    
    /**
     * @var \Webkul\AdvancedBookingSystem\Model\QuestionFactory
     */
    private $questionModel;
    
    /**
     * @var \Magento\Framework\Stdlib\DateTime\DateTime
     */
    private $date;
    
    /**
     * @var \Webkul\AdvancedBookingSystem\Helper\Email
     */
    private $emailHelper;

    /**
     *
     * @param Data $dataHelper
     * @param \Webkul\AdvancedBookingSystem\Model\QuestionFactory $questionModel
     * @param \Magento\Framework\Stdlib\DateTime\DateTime $date
     * @param \Webkul\AdvancedBookingSystem\Helper\Email $emailHelper
     */
    public function __construct(
        Data $dataHelper,
        \Webkul\AdvancedBookingSystem\Model\QuestionFactory $questionModel,
        \Magento\Framework\Stdlib\DateTime\DateTime $date,
        \Webkul\AdvancedBookingSystem\Helper\Email $emailHelper
    ) {
        $this->dataHelper = $dataHelper;
        $this->questionModel = $questionModel;
        $this->date = $date;
        $this->emailHelper = $emailHelper;
    }

    /**
     * SubmitQuestion Resolver
     *
     * @param \Magento\Framework\GraphQl\Config\Element\Field $field
     * @param ContextInterface $context
     * @param ResolveInfo $info
     * @param array|null $value
     * @param array|null $args
     * @throws GraphQlException
     * @return string
     */
    public function resolve(
        Field $field,
        $context,
        ResolveInfo $info,
        array $value = null,
        array $args = null
    ) {
        try {
            if (!isset($args['productId'])) {
                throw new GraphQlInputException(__('Product Id is required.'));
            }
            $productId = $args['productId'];
            $userId = $context->getUserId();
            $helper = $this->dataHelper;
            $product = $helper->getProduct($productId);
            if (!$product) {
                throw new GraphQlNoSuchEntityException(__('Product doesn\'t exist'));
            }
            $productType = $product->getTypeId();
            $productSetId = $product->getAttributeSetId();
            $hotelAttrSetId = $helper->getProductAttributeSetIdByLabel(
                'Hotel Booking'
            );
            if ($productType != 'hotelbooking' || $productSetId != $hotelAttrSetId) {
                throw new GraphQlNoSuchEntityException(__('Not a Hotel Booking Type Product'));
            }
            if (!$product['ask_a_ques_enable']) {
                throw new GraphQlInputException(__('Ask a question is not enabled for this product'));
            }
            if (!isset($args['customerNickname']) && $args['customerNickname'] == "") {
                throw new GraphQlInputException(__('Please enter your nick name'));
            } elseif (!isset($args['userQuestion']) && $args['userQuestion'] == "") {
                throw new GraphQlInputException(__('Please enter your question'));
            }
            $data['product_id'] = $productId;
            $data['product_name'] = $product->getName();
            $data['nick_name'] = $args['customerNickname'];
            $data['question'] = $args['userQuestion'];
            $data['customer_id'] = $userId ? $userId : 0;
            $data['created_at'] = $this->date->gmtDate();
            $data['updated_at'] = $this->date->gmtDate();
            if (!$helper->getConfigValue('auto_approve_question')) {
                $data['status'] = \Webkul\AdvancedBookingSystem\Model\Question::STATUS_APPROVED;
            }
            $model = $this->questionModel->create()->setData($data)->save();
            $id = $model->getId();
            if ($id) {
                $emailTemplateVariables = [];
                $emailTemplateVariables['myvar1'] = 'Admin';
                if (!empty($data['product_id'])) {
                    $product = $helper->getProduct($data['product_id']);
                    $emailTemplateVariables['product_name'] = $product->getName();
                    $emailTemplateVariables['product_sku'] = $product->getSku();
                } else {
                    $emailTemplateVariables['product_name'] = "hotel";
                    $emailTemplateVariables['product_sku'] = "hotel";
                }
                $emailTemplateVariables['question'] = $data['question'];
                $emailTemplateVariables['subject'] = __(
                    "On booking of %1",
                    $emailTemplateVariables['product_name']
                );
                $emailTemplateVariables['subject'] = (string)$emailTemplateVariables['subject'];

                $this->emailHelper->sendQuestionMailToAdmin(
                    $data,
                    $emailTemplateVariables
                );
                return (
                    __('Your question has been submitted successfully')
                );
            } else {
                return (
                    __('Something went wrong !!!')
                );
            }
        } catch (LocalizedException $e) {
            throw new GraphQlNoSuchEntityException(__($e->getMessage()));
        }
    }
}
