<?php

declare(strict_types=1);

/**
 * Webkul Software.
 *
 * @category  Webkul
 * @package   Webkul_AdvancedBookingSystem
 * @author    Webkul Software Private Limited
 * @copyright Webkul Software Private Limited (https://webkul.com)
 * @license   https://store.webkul.com/license.html
 */

namespace Webkul\AdvancedBookingSystem\Model\Resolver;

use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlAuthorizationException;
use Magento\Framework\GraphQl\Exception\GraphQlNoSuchEntityException;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Webkul\AdvancedBookingSystem\Helper\Data;

/**
 * Resolver for TableBookingInfo query
 */
class TableBookingInfo implements ResolverInterface
{
    /**
     * @var Data
     */
    private $dataHelper;

    /**
     *
     * @param Data $dataHelper
     */
    public function __construct(
        Data $dataHelper,
    ) {
        $this->dataHelper = $dataHelper;
    }

    /**
     * TableBookingInfo Resolver
     *
     * @param \Magento\Framework\GraphQl\Config\Element\Field $field
     * @param ContextInterface $context
     * @param ResolveInfo $info
     * @param array|null $value
     * @param array|null $args
     * @throws GraphQlException
     * @return array
     */
    public function resolve(
        Field $field,
        $context,
        ResolveInfo $info,
        array $value = null,
        array $args = null
    ) {
        try {
            $productId = $args['productId'];
            $tableBookingInfo = [];
            $helper = $this->dataHelper;
            $product = $helper->getProduct($productId);
            if (!$product) {
                throw new GraphQlNoSuchEntityException(__('Product doesn\'t exist'));
            }
            $productType = $product->getTypeId();
            $productSetId = $product->getAttributeSetId();
            $tableAttrSetId = $helper->getProductAttributeSetIdByLabel(
                'Table Booking'
            );
            if ($productType != 'booking' || $productSetId != $tableAttrSetId) {
                throw new GraphQlNoSuchEntityException(__('Not a Table Type Product'));
            }
            $tableBookingInfo['available'] = 1;
            $tableBookingInfo['name'] = $product->getName();
            $tableBookingInfo['sku'] = $product->getSku();
            $tableBookingInfo['productPrice'] = $product->getFinalPrice();
            $currentLocalTime = $helper->getCurrentTime();
            $currentDate = strtotime($helper->getCurrentDate());
            $bookingAvailableFrom = date('m/d/Y');
            $bookingAvailableTo = '';
            $bookingInfo = $helper->getBookingInfo($productId);
            $bookingSlotData = $helper->getJsonDecodedString($bookingInfo['info']);
            $bookedData = $helper->getBookedAppointmentDates($productId);
            // Calculated current day slot data
            $today = date("l");
            $todayIndex = $helper->getDayIndexId($today);
            if (empty($bookingSlotData[1])) {
                $bookingSlotData[1] = [];
            }
            if (empty($bookingSlotData[2])) {
                $bookingSlotData[2] = [];
            }
            if (empty($bookingSlotData[3])) {
                $bookingSlotData[3] = [];
            }
            if (empty($bookingSlotData[4])) {
                $bookingSlotData[4] = [];
            }
            if (empty($bookingSlotData[5])) {
                $bookingSlotData[5] = [];
            }
            if (empty($bookingSlotData[6])) {
                $bookingSlotData[6] = [];
            }
            if (empty($bookingSlotData[7])) {
                $bookingSlotData[7] = [];
            }
            $guestsCapacityArr = $helper->getMaxGuestsAvailable();
            $maxCapacity = 1;
            if ($product['max_capacity']) {
                $maxCapacity = $product['max_capacity'];
            }
            $minNoOfGuestsAllowed = 1;
            if (!empty($product['no_of_guests']) && $product['no_of_guests'] > 0) {
                $minNoOfGuestsAllowed = $product['no_of_guests'];
            }
            $customOptChargedPerValue = "";
            if (!empty($product['price_charged_per_table']) && $product['price_charged_per_table'] == 1) {
                $customOptChargedPerValue = "Guest";
            }
            $customOptionLabel = __("Number of Guests");
            if (!empty($product['no_of_guests']) && $product['no_of_guests'] > 1) {
                if ($product['price_charged_per_table'] == 1) {
                    $customOptChargedPerValue = "(" . $product['no_of_guests'] . " " . $customOptChargedPerValue . " )";
                    $customOptionLabel = __("Number of Guests");
                } else {
                    $customOptChargedPerValue .= "Table (" . $product['no_of_guests'] . " Guests)";
                    $maxCapacity = (int)($maxCapacity / $product['no_of_guests']);
                    $customOptionLabel = __("Number of Table");
                }
            }
            $tableBookingInfo['minNoOfGuestsAllowed'] = $minNoOfGuestsAllowed;
            $tableBookingInfo['maxCapacity'] = $maxCapacity;
            $tableBookingInfo['chargePerLabel'] = $customOptionLabel;
            $tableBookingInfo['chargePerValue'] = $customOptChargedPerValue;
            $stateName = $helper->getRegionName($product['hotel_state'], $product['hotel_country']);
            $formattedAddress = $product['hotel_address'] . ", " . $product['location']
                . $stateName . ", " . $product['hotel_country'];
            $tableBookingInfo['location'] = $formattedAddress;
            $tableBookingInfo['slotDuration'] = $product['slot_duration'];
            $tableBookingInfo['showMapLoction'] = $product['show_map_loction'];
            $tableBookingInfo['showContactButtonTo'] = ($product['show_contact_button_to'] == 0) ?
            'Everyone' : 'Logged In Customer';
            $tableBookingInfo['availabilityLabel'] = $product['slot_for_all_days']==1 ? 'Everyday' : 'Today';
            $todaySlotsCount = 0;
            $todaySlots = [];
            $isTodayHoliday = $helper->isTodayHoliday($productId);
            if (isset($bookingSlotData[$todayIndex]) && !$isTodayHoliday) {
                $todaySlots = $bookingSlotData[$todayIndex];
                $todaySlotsCount = count($todaySlots);
            }
            $closed = ($currentDate >= strtotime($bookingAvailableFrom)) && $todaySlotsCount ? false : true;
            $tableBookingInfo['available'] = !$closed;
            $tableBookingInfo['todaySlots'] = $todaySlots;
            return $tableBookingInfo;
        } catch (LocalizedException $e) {
            throw new GraphQlNoSuchEntityException(__($e->getMessage()));
        }
    }
}
