<?php
/**
 * Webkul Software.
 *
 * @category  Webkul
 * @package   Webkul_AdvancedBookingSystem
 * @author    Webkul Software Private Limited
 * @copyright Webkul Software Private Limited (https://webkul.com)
 * @license   https://store.webkul.com/license.html
 */
namespace Webkul\AdvancedBookingSystem\Model\Sales\Order\Email;

use Magento\Sales\Model\Order\Email\Container\Template;
use Magento\Sales\Model\Order\Email\Container\IdentityInterface;
use Magento\Framework\Mail\Template\TransportBuilder;
use Magento\Framework\Mail\Template\TransportBuilderByStore;

/**
 * Email sender builder for attachments
 */
class SenderBuilder extends \Magento\Sales\Model\Order\Email\SenderBuilder
{
    /** @var \Webkul\AdvancedBookingSystem\Helper\Data */
    protected $helper;
    
    /** @var \Magento\Framework\Filesystem\Driver\File */
    protected $reader;

    /**
     * @param Template $templateContainer
     * @param IdentityInterface $identityContainer
     * @param TransportBuilder $transportBuilder
     * @param TransportBuilderByStore $transportBuilderByStore
     * @param \Webkul\AdvancedBookingSystem\Helper\Data $helper
     * @param \Magento\Framework\Filesystem\Driver\File $reader
     */
    public function __construct(
        Template $templateContainer,
        IdentityInterface $identityContainer,
        TransportBuilder $transportBuilder,
        TransportBuilderByStore $transportBuilderByStore = null,
        \Webkul\AdvancedBookingSystem\Helper\Data $helper,
        \Magento\Framework\Filesystem\Driver\File $reader
    ) {
        parent::__construct(
            $templateContainer,
            $identityContainer,
            $transportBuilder
        );
        $this->helper = $helper;
        $this->reader = $reader;
    }

    /**
     * Prepare and send email message
     *
     * @return void
     */
    public function send()
    {
        $this->configureEmailTemplate();

        $this->transportBuilder->addTo(
            $this->identityContainer->getCustomerEmail(),
            $this->identityContainer->getCustomerName()
        );

        $copyTo = $this->identityContainer->getEmailCopyTo();

        if (!empty($copyTo) && $this->identityContainer->getCopyMethod() == 'bcc') {
            foreach ($copyTo as $email) {
                $this->transportBuilder->addBcc($email);
            }
        }

        /* to attach events in invoice email */
        $templateVars = $this->templateContainer->getTemplateVars();
        $transport = $this->transportBuilder->getTransport();

        if (!empty($templateVars['order'])) {
            $order = $templateVars['order'];
            foreach ($order->getAllItems() as $item) {
                $productId = $item->getProductId();
                if ($this->helper->isBookingProduct($productId)) {
                    $data = $this->helper->createIcsFile($item, $order->getId());
                    if (!empty($data) && !empty($data['filename']) && !empty($data['icsFile'])) {
                        // adds attachment in mail
                        $attachmentPart = $this->transportBuilder->addAttachment(
                            $this->reader->fileGetContents($data['icsFile']),
                            $data['filename'],
                            'text/calendar'
                        );

                        $message = $transport->getMessage();
                        $body = \Zend\Mail\Message::fromString($message->getRawMessage())->getBody();
                        $body = \Laminas\Mime\Decode::decodeQuotedPrintable($body);
                        $html = '';

                        if ($body instanceof \Zend\Mime\Message) {
                            $html = $body->generateMessage(\Zend\Mail\Headers::EOL);
                        } elseif ($body instanceof \Magento\Framework\Mail\MimeMessage) {
                            $html = (string) $body->getMessage();
                        } elseif ($body instanceof \Magento\Framework\Mail\EmailMessage) {
                            $html = (string) $body->getBodyText();
                        } else {
                            $html = (string) $body;
                        }

                        $htmlPart = new \Zend\Mime\Part($html);
                        $htmlPart->setCharset('utf-8');
                        $htmlPart->setEncoding(\Laminas\Mime\Mime::ENCODING_QUOTEDPRINTABLE);
                        $htmlPart->setDisposition(\Laminas\Mime\Mime::DISPOSITION_INLINE);
                        $htmlPart->setType(\Laminas\Mime\Mime::TYPE_HTML);
                        $parts = [$htmlPart, $attachmentPart];

                        $bodyPart = new \Zend\Mime\Message();
                        $bodyPart->setParts($parts);
                        $message->setBody($bodyPart);
                    }
                }
            }
        }
        
        $transport->sendMessage();
    }
}
