<?php

/**
 * Webkul Software.
 *
 * @category  Webkul
 * @package   Webkul_AdvancedBookingSystem
 * @author    Webkul Software Private Limited
 * @copyright Webkul Software Private Limited (https://webkul.com)
 * @license   https://store.webkul.com/license.html
 */

namespace Webkul\AdvancedBookingSystem\Observer;

use Magento\Framework\Event\ObserverInterface;
use Magento\Framework\App\Filesystem\DirectoryList;

class AfterProductSave implements ObserverInterface
{
    /**
     * @var \Magento\Framework\App\RequestInterface
     */
    protected $request;

    /**
     * @var \Webkul\AdvancedBookingSystem\Helper\Data
     */
    protected $helper;

    /**
     * @var \Webkul\AdvancedBookingSystem\Model\InfoFactory
     */
    protected $info;

    /**
     * @var \Webkul\AdvancedBookingSystem\Model\SlotFactory
     */
    protected $slot;

    /**
     * @var \Magento\ConfigurableProduct\Model\ResourceModel\Product\Type\Configurable
     */
    protected $configurable;

    /**
     * @var \Webkul\AdvancedBookingSystem\Model\ResourceModel\Info\CollectionFactory
     */
    protected $infoCollection;

    /**
     * @var \Magento\InventoryCatalogApi\Model\SourceItemsProcessorInterface
     */
    private $sourceItemsProcessor;

    /**
     * @var \Magento\InventoryCatalogApi\Model\GetSkusByProductIdsInterface
     */
    private $getSkusByProductIdsInterface;

    /**
     * @var \Magento\Framework\Filesystem
     */
    private $filesystem;

    /**
     * @var \Magento\Framework\Filesystem\Driver\File
     */
    private $filesystemDriverFile;

    /**
     * @var \Magento\Framework\Filesystem\Driver\File
     */
    private $mediaDirectory;

    /**
     * @var \Magento\MediaStorage\Model\File\UploaderFactory
     */
    private $fileUploaderFactory;

    /**
     * Constructor classes for AfterProductSave
     *
     * @param \Magento\Framework\App\RequestInterface $request
     * @param \Webkul\AdvancedBookingSystem\Helper\Data $helper
     * @param \Webkul\AdvancedBookingSystem\Model\InfoFactory $info
     * @param \Webkul\AdvancedBookingSystem\Model\SlotFactory $slot
     * @param \Magento\ConfigurableProduct\Model\ResourceModel\Product\Type\Configurable $configurable
     * @param \Webkul\AdvancedBookingSystem\Model\ResourceModel\Info\CollectionFactory $infoCollectionFactory
     * @param \Magento\Framework\Filesystem $filesystem
     * @param \Magento\Framework\Filesystem\Driver\File $filesystemDriverFile
     * @param \Magento\MediaStorage\Model\File\UploaderFactory $fileUploaderFactory
     * @param \Magento\InventoryCatalogApi\Model\SourceItemsProcessorInterface $sourceItemsProcessor
     * @param \Magento\InventoryCatalogApi\Model\GetSkusByProductIdsInterface $getSkusByProductIdsInterface
     */
    public function __construct(
        \Magento\Framework\App\RequestInterface $request,
        \Webkul\AdvancedBookingSystem\Helper\Data $helper,
        \Webkul\AdvancedBookingSystem\Model\InfoFactory $info,
        \Webkul\AdvancedBookingSystem\Model\SlotFactory $slot,
        \Magento\ConfigurableProduct\Model\ResourceModel\Product\Type\Configurable $configurable,
        \Webkul\AdvancedBookingSystem\Model\ResourceModel\Info\CollectionFactory $infoCollectionFactory,
        \Magento\Framework\Filesystem $filesystem,
        \Magento\Framework\Filesystem\Driver\File $filesystemDriverFile,
        \Magento\MediaStorage\Model\File\UploaderFactory $fileUploaderFactory,
        \Magento\InventoryCatalogApi\Model\SourceItemsProcessorInterface $sourceItemsProcessor,
        \Magento\InventoryCatalogApi\Model\GetSkusByProductIdsInterface $getSkusByProductIdsInterface
    ) {
        $this->request = $request;
        $this->helper = $helper;
        $this->info = $info;
        $this->slot = $slot;
        $this->configurable = $configurable;
        $this->infoCollection = $infoCollectionFactory;
        $this->filesystem = $filesystem;
        $this->filesystemDriverFile = $filesystemDriverFile;
        $this->mediaDirectory = $filesystem->getDirectoryWrite(
            DirectoryList::MEDIA
        );
        $this->fileUploaderFactory = $fileUploaderFactory;
        $this->sourceItemsProcessor = $sourceItemsProcessor;
        $this->getSkusByProductIdsInterface = $getSkusByProductIdsInterface;
    }

    /**
     * Save booking product records while saving product
     *
     * @param \Magento\Framework\Event\Observer $observer
     * @return void
     */
    public function execute(\Magento\Framework\Event\Observer $observer)
    {
        try {
            $product = $observer->getEvent()->getProduct();
            $productId = $product->getId();
            $productType = $product->getTypeId();
            $data = $this->request->getParams();
            $files = $this->request->getFiles();

            if ($productType == "virtual") {
                $this->checkHotelChildProduct($productId, $data);
            }
            if ($productType != "booking" && $productType != "hotelbooking") {
                return;
            }
            $productSetId = $product->getAttributeSetId();
            $infoModel = $this->info->create();
            $slotModel = $this->slot->create();
            $helper = $this->helper;

            $appointmentAttrSetId = $helper->getProductAttributeSetIdByLabel(
                'Appointment Booking'
            );
            $eventAttrSetId = $helper->getProductAttributeSetIdByLabel(
                'Event Booking'
            );
            $rentalAttrSetId = $helper->getProductAttributeSetIdByLabel(
                'Rental Booking'
            );
            $hotelAttrSetId = $helper->getProductAttributeSetIdByLabel(
                'Hotel Booking'
            );
            $tableAttrSetId = $helper->getProductAttributeSetIdByLabel(
                'Table Booking'
            );
            $allowedAttrSetIDs = $helper->getAllowedAttrSetIDs();
            if ($productSetId == $appointmentAttrSetId) {
                $this->saveAppointmentBooking($data, $productId, $productSetId);
            } elseif ($productSetId == $eventAttrSetId) {
                $this->saveEventBooking($product, $data, $productId, $productSetId);
            } elseif ($productSetId == $rentalAttrSetId) {
                $this->saveRentalBooking($product, $data, $productId, $productSetId);
            } elseif ($productSetId == $hotelAttrSetId) {
                $configurableMatrix = $this->request->getParam('configurable-matrix-serialized', '');
                $this->processSourceItems($product, $configurableMatrix);
                $this->saveHotelBooking($data, $productId, $productSetId, $files);
            } elseif ($productSetId == $tableAttrSetId) {
                $this->saveTableBooking($data, $productId, $productSetId);
            } elseif (!in_array($productSetId, $allowedAttrSetIDs)) {
                $this->saveDefaultBooking($data, $productId, $productSetId);
            }
            $this->helper->enableOptions($productId);
            $this->helper->checkBookingProduct($productId);
        } catch (\Exception $e) {
            $this->helper->logDataInLogger(
                "Observer_AfterProductSave execute : " . $e->getMessage()
            );
        }
    }

    /**
     * Save Appointment Booking Product.
     *
     * @param array $data
     * @param int $productId
     * @param int $productSetId
     * @return void
     */
    public function saveAppointmentBooking($data, $productId, $productSetId)
    {
        try {
            $helper = $this->helper;
            $infoModel = $this->info->create();
            $slotModel = $this->slot->create();
            if (empty($data['product']['slot_data'])) {
                return false;
            }
            $data['product'] = $this->getCalculatedSlotData($data['product']);
            $totalSlots = $data['product']['total_slots'];
            $qty = 0;

            if (isset($data['product']['quantity_and_stock_status']['qty'])) {
                $qty = $data['product']['quantity_and_stock_status']['qty'];
            } elseif (isset($data['product']['stock_data']['qty'])) {
                $qty = $data['product']['stock_data']['qty'];
            }

            if ($data['product']['slot_for_all_days']) {
                $data['product']['slot_data'][2] = $data['product']['slot_data'][1];
                $data['product']['slot_data'][3] = $data['product']['slot_data'][1];
                $data['product']['slot_data'][4] = $data['product']['slot_data'][1];
                $data['product']['slot_data'][5] = $data['product']['slot_data'][1];
                $data['product']['slot_data'][6] = $data['product']['slot_data'][1];
                $data['product']['slot_data'][7] = $data['product']['slot_data'][1];
            }
            $slotDuration = $data['product']['slot_duration'];
            $slotBreakTime = $data['product']['break_time_bw_slot'];
            $totalSlotTime = $slotDuration + $slotBreakTime;
            foreach ($data['product']['slot_data'] as $dayKey => $dayValue) {
                $slotIndex = 1;
                $currentSlots = 0;
                if (!empty($dayValue)) {
                    foreach ($dayValue as $key => $value) {
                        if (empty($value['qty'])) {
                            $value['qty'] = 1;
                        }
                        if (!$data['product']['slot_has_quantity']) {
                            $value['qty'] = 1;
                        }
                        $availableSlotArr = [];
                        $slotFromTimeStamp = strtotime($value['from']);
                        $slotToTimeStamp = strtotime($value['to']);
                        $totalTimeStamp = ($slotToTimeStamp - $slotFromTimeStamp) / 60; // in hours
                        $currentSlots = $currentSlots + (int)($totalTimeStamp / $totalSlotTime);
                        $availableSlotArr[$slotIndex - 1]['qty'] = $value['qty'];
                        $availableSlotArr[$slotIndex - 1]['time'] = $value['from'];
                        for ($i = $slotIndex; $i < $currentSlots; $i++) {
                            $availableSlotArr[$i]['qty'] = $value['qty'];
                            $availableSlotArr[$i]['time'] = date(
                                "h:i a",
                                strtotime(
                                    '+' . $totalSlotTime . ' minutes',
                                    strtotime($availableSlotArr[$i - 1]['time'])
                                )
                            );
                        }
                        $slotIndex = $currentSlots;
                        $slotIndex++;
                        $data['product']['slot_data'][$dayKey][$key]['qty'] = $value['qty'];
                        $data['product']['slot_data'][$dayKey][$key]['slots_info'] = $availableSlotArr;
                    }
                }
            }
            $bookingSlotData = $helper->getJsonEcodedString($data['product']['slot_data']);

            $slotProductId = 0;
            $slotCollection = $slotModel->getCollection()
                ->addFieldToFilter("product_id", $productId);
            foreach ($slotCollection as $value) {
                $slotProductId = $value->getId();
            }

            $infoModelOld = null;
            $bookingProductId = 0;
            $bookingCollection = $infoModel->getCollection()
                ->addFieldToFilter("product_id", $productId);
            foreach ($bookingCollection as $value) {
                $bookingProductId = $value->getId();
                $infoModelOld = $value;
            }

            $type = $data['product']['slot_for_all_days'];
            if ($this->canSaveAppointmentSlots(
                $infoModelOld,
                $bookingSlotData,
                $type,
                $totalSlots,
                $data['product']['booking_available_from'],
                $data['product']['booking_available_to']
            )) {
                $infoData = [
                    'attribute_set_id' => $productSetId,
                    'product_id' => $productId,
                    'type' => $type,
                    'start_date' => $data['product']['booking_available_from'],
                    'end_date' => $data['product']['booking_available_to'],
                    'info' => $bookingSlotData,
                    'qty' => $qty,
                    'total_slots' => $totalSlots
                ];
                $slotData = [
                    'product_id' => $productId,
                    'type' => $type,
                    'status' => 1
                ];
                $slotModel->load($slotProductId)
                    ->setData($slotData)
                    ->save();

                $infoModel->load($bookingProductId)
                    ->addData($infoData)
                    ->save();
            }
        } catch (\Exception $e) {
            $this->helper->logDataInLogger($e->getMessage());
        }
    }

    /**
     * Can Save Appointment Booking Product.
     *
     * @param object $infoModelOld
     * @param string $newJsonData
     * @param int $type
     * @param int $totalSlots
     * @param string $startDate
     * @param string $endDate
     * @return bool
     */
    public function canSaveAppointmentSlots($infoModelOld, $newJsonData, $type, $totalSlots, $startDate, $endDate)
    {
        if (empty($infoModelOld) || $infoModelOld->getType() != $type) {
            return true;
        }

        //if total slots value changed
        if ($infoModelOld->getTotalSlots() != $totalSlots) {
            return true;
        }

        //if total slots value changed
        if ($infoModelOld->getStartDate() != $startDate || $infoModelOld->getEndDate() != $endDate) {
            return true;
        }

        $oldJson = $this->helper->getJsonDecodedString($infoModelOld->getInfo());
        $newJson = $this->helper->getJsonDecodedString($newJsonData);

        if ($this->checkAppointmentSlotsData($oldJson, $newJson)) {
            return true;
        }
    }

    /**
     * Check Appointment Booking Product.
     *
     * @param array $oldSlots
     * @param array $newSlots
     *
     * @return bool
     */
    public function checkAppointmentSlotsData($oldSlots, $newSlots)
    {
        if (count($oldSlots) !== count($newSlots)) {
            return true;
        }

        foreach ($newSlots as $key => $daysData) {
            if (!isset($oldSlots[$key]) || count($oldSlots[$key]) !== count($daysData)) {
                return true;
            }

            foreach ($daysData as $slotKey => $slotsData) {
                if (!isset($oldSlots[$key][$slotKey])) {
                    return true;
                }
                $oldSlotData = $oldSlots[$key][$slotKey];
                if (isset($oldSlotData['from']) && $oldSlotData['from'] !== $slotsData['from']) {
                    return true;
                }

                if (isset($oldSlotData['to']) && $oldSlotData['to'] !== $slotsData['to']) {
                    return true;
                }

                if (isset($slotsData['slots_info']) && isset($oldSlotData['slots_info'])) {
                    $oldSlotInfo = $oldSlotData['slots_info'];
                    foreach ($slotsData['slots_info'] as $slotInfoKey => $slotInfo) {
                        if (isset($oldSlotInfo[$slotInfoKey]['qty'])
                            && $oldSlotInfo[$slotInfoKey]['qty'] !== $slotInfo['qty']
                        ) {
                            return true;
                        }

                        if (isset($oldSlotInfo[$slotInfoKey]['time'])
                            && $oldSlotInfo[$slotInfoKey]['time'] !== $slotInfo['time']
                        ) {
                            return true;
                        }
                    }
                } else {
                    return true;
                }
            }
        }
    }

    /**
     * Save Event Booking Product.
     *
     * @param \Magento\Catalog\Model\Product $product
     * @param array $data
     * @param int $productId
     * @param int $productSetId
     * @return void
     */
    public function saveEventBooking($product, $data, $productId, $productSetId)
    {
        try {
            $helper = $this->helper;
            $infoModel = $this->info->create();
            $slotModel = $this->slot->create();
            if (empty($data['product']['options'])) {
                return false;
            }
            $eventOptions = $data['product']['options'];
            $optionIndex = 0;
            foreach ($eventOptions as $index => $option) {
                if ($option['title'] == 'Event Tickets') {
                    $optionIndex = $index;
                    break;
                }
            }
            $totalSlots = 0;
            $qty = 0;

            if (isset($data['product']['quantity_and_stock_status']['qty'])) {
                $qty = $data['product']['quantity_and_stock_status']['qty'];
            } elseif (isset($data['product']['stock_data']['qty'])) {
                $qty = $data['product']['stock_data']['qty'];
            }

            if (empty($eventOptions[$optionIndex]['values'])) {
                return false;
            }
            foreach ($eventOptions[$optionIndex]['values'] as $optionValue) {
                $totalSlots = $totalSlots + (int)$optionValue['qty'];
            }

            $updatedBookingOptions = [];
            $customOptions = $product->getOptions();
            $index = 0;
            foreach ($product->getProductOptionsCollection() as $key => $customOption) {
                $customOptionData = $customOption->getData();
                if (!empty($customOptionData['title']) && $customOptionData['title'] == 'Event Tickets') {
                    $updatedBookingOptions[$index] = $customOptionData;
                    $customOptionValues = $customOption->getValues() ?? [];
                    $optionValuesData = [];
                    foreach ($customOptionValues as $customOptionValue) {
                        $optionValuesData[] = $customOptionValue->getData();
                    }
                    $updatedBookingOptions[$index]['values'] = $optionValuesData;
                }
                $index++;
            }

            if (!empty($updatedBookingOptions)) {
                $bookingOptionsData = $helper->getJsonEcodedString($updatedBookingOptions);
            } else {
                $bookingOptionsData = $helper->getJsonEcodedString($eventOptions);
            }

            $slotProductId = 0;
            $slotCollection = $slotModel->getCollection()
                ->addFieldToFilter("product_id", $productId);
            foreach ($slotCollection as $value) {
                $slotProductId = $value->getId();
            }
            $slotData = [
                'product_id' => $productId,
                'type' => 1,
                'status' => 1
            ];
            $slotModel->load($slotProductId)
                ->setData($slotData)
                ->save();
            $bookingProductId = 0;
            $bookingCollection = $infoModel->getCollection()
                ->addFieldToFilter("product_id", $productId);
            foreach ($bookingCollection as $value) {
                $bookingProductId = $value->getId();
            }
            $fromTime = date(
                'Y-m-d H:i',
                strtotime($data['product']['event_date_from'])
            );
            $toTime = date(
                'Y-m-d H:i',
                strtotime($data['product']['event_date_to'])
            );

            $infoData = [
                'attribute_set_id' => $productSetId,
                'product_id' => $productId,
                'type' => 1,
                'start_date' => $fromTime,
                'end_date' => $toTime,
                'info' => $bookingOptionsData,
                'qty' => $qty,
                'total_slots' => $totalSlots
            ];
            $infoModel->load($bookingProductId)
                ->addData($infoData)
                ->save();
        } catch (\Exception $e) {
            $this->helper->logDataInLogger($e->getMessage());
        }
    }

    /**
     * Save Rental Booking Product.
     *
     * @param \Magento\Catalog\Model\Product $product
     * @param array $data
     * @param int $productId
     * @param int $productSetId
     * @return void
     */
    public function saveRentalBooking($product, $data, $productId, $productSetId)
    {
        try {
            $helper = $this->helper;
            $infoModel = $this->info->create();
            $slotModel = $this->slot->create();
            if (empty($data['product']['renting_type'])) {
                return false;
            }
            if (empty($data['product']['options'])) {
                return false;
            }
            $rentType = $data['product']['renting_type'];
            $qty = 0;

            if (isset($data['product']['quantity_and_stock_status']['qty'])) {
                $qty = $data['product']['quantity_and_stock_status']['qty'];
            } elseif (isset($data['product']['stock_data']['qty'])) {
                $qty = $data['product']['stock_data']['qty'];
            }

            $availableQty = $qty;
            if (!empty($data['product']['available_qty']) && $data['product']['available_qty']) {
                $availableQty = $data['product']['available_qty'];
            }
            if ($rentType != 1) {
                if (empty($data['product']['slot_data'])) {
                    return false;
                }
                $data['product'] = $this->getCalculatedRentSlotData($data['product']);
                $totalSlots = $data['product']['total_slots'];
                $totalSlotTime = 60; //60 mins
                foreach ($data['product']['slot_data'] as $dayKey => $dayValue) {
                    $slotIndex = 1;
                    $currentSlots = 0;
                    if (empty($dayValue)) {
                        continue;
                    }
                    foreach ($dayValue as $key => $value) {
                        if (empty($value['qty'])) {
                            $value['qty'] = $availableQty;
                        }
                        if (!$data['product']['slot_has_quantity']) {
                            $value['qty'] = $availableQty;
                        }
                        $availableSlotArr = [];
                        $slotFromTimeStamp = strtotime($value['from']);
                        $slotToTimeStamp = strtotime($value['to']);
                        $totalTimeStamp = ($slotToTimeStamp - $slotFromTimeStamp) / 60; // in hours
                        $currentSlots = $currentSlots + (int)($totalTimeStamp / $totalSlotTime);
                        $availableSlotArr[$slotIndex - 1]['qty'] = $value['qty'];
                        $availableSlotArr[$slotIndex - 1]['time'] = $value['from'];
                        for ($i = $slotIndex; $i <= $currentSlots; $i++) {
                            $availableSlotArr[$i]['qty'] = $value['qty'];
                            $availableSlotArr[$i]['time'] = date(
                                "h:i a",
                                strtotime(
                                    '+' . $totalSlotTime . ' minutes',
                                    strtotime($availableSlotArr[$i - 1]['time'])
                                )
                            );
                        }
                        $slotIndex = $currentSlots;
                        $slotIndex++;
                        $data['product']['slot_data'][$dayKey][$key]['qty'] = $value['qty'];
                        $data['product']['slot_data'][$dayKey][$key]['slots_info'] = $availableSlotArr;
                    }
                }
                if ($data['product']['slot_for_all_days']) {
                    $data['product']['slot_data'][2] = $data['product']['slot_data'][1];
                    $data['product']['slot_data'][3] = $data['product']['slot_data'][1];
                    $data['product']['slot_data'][4] = $data['product']['slot_data'][1];
                    $data['product']['slot_data'][5] = $data['product']['slot_data'][1];
                    $data['product']['slot_data'][6] = $data['product']['slot_data'][1];
                    $data['product']['slot_data'][7] = $data['product']['slot_data'][1];
                }
                $bookingSlotData = $helper->getJsonEcodedString($data['product']['slot_data']);
            } else {
                if ($data['product']['available_every_week']) {
                    $totalSlots = 99999999;
                } else {
                    $startDate = $data['product']['booking_available_from'];
                    $endDate = $data['product']['booking_available_to'];
                    $numOfDays = $this->getDateDifference($startDate, $endDate);
                    $numOfDays++;
                    $totalSlots = $availableQty * $numOfDays;
                }
                $bookingSlotData = $helper->getJsonEcodedString([]);
            }
            $slotProductId = 0;
            $slotCollection = $slotModel->getCollection()
                ->addFieldToFilter("product_id", $productId);
            foreach ($slotCollection as $value) {
                $slotProductId = $value->getId();
            }

            $bookingProductId = 0;
            $infoModelOld = null;
            $bookingCollection = $infoModel->getCollection()
                ->addFieldToFilter("product_id", $productId);
            foreach ($bookingCollection as $value) {
                $bookingProductId = $value->getId();
                $infoModelOld = $value;
            }
            if ($this->canSaveRentalSlots(
                $infoModelOld,
                $bookingSlotData,
                $rentType,
                $data['product']['booking_available_from'],
                $data['product']['booking_available_to'],
                $availableQty
            )) {
                $infoData = [
                    'attribute_set_id' => $productSetId,
                    'product_id' => $productId,
                    'type' => $rentType,
                    'start_date' => $data['product']['booking_available_from'],
                    'end_date' => $data['product']['booking_available_to'],
                    'info' => $bookingSlotData,
                    'qty' => $qty,
                    'available_qty' => $availableQty,
                    'total_slots' => $totalSlots
                ];
                $slotData = [
                    'product_id' => $productId,
                    'type' => $rentType,
                    'status' => 1
                ];
                $slotModel->load($slotProductId)
                    ->setData($slotData)
                    ->save();

                $infoModel->load($bookingProductId)
                    ->addData($infoData)
                    ->save();
            }
        } catch (\Exception $e) {
            $this->helper->logDataInLogger($e->getMessage());
        }
    }

    /**
     * Check Whether New Slots or Not
     *
     * @param array  $infoModelOld
     * @param string $newJsonData
     * @param int    $type
     * @param string $dateFrom
     * @param string $dateTo
     * @param int    $availableQty
     *
     * @return bool
     */
    public function canSaveRentalSlots($infoModelOld, $newJsonData, $type, $dateFrom, $dateTo, $availableQty)
    {
        if ($infoModelOld === null) {
            return true;
        }

        if ($infoModelOld->getAvailableQty() != $availableQty) {
            return true;
        }

        if ($infoModelOld->getType() != $type) {
            return true;
        }

        if ($infoModelOld->getStartDate() != $dateFrom || $infoModelOld->getEndDate() != $dateTo) {
            return true;
        }

        $oldJson = $this->helper->getJsonDecodedString($infoModelOld->getInfo());
        $newJson = $this->helper->getJsonDecodedString($newJsonData);
        if ($this->checkRentalSlotsData($oldJson, $newJson)) {
            return true;
        }
    }

    /**
     * Check Rental Slot Data
     *
     * @param array $oldSlots
     * @param array $newSlots
     *
     * @return bool
     */
    public function checkRentalSlotsData($oldSlots, $newSlots)
    {
        foreach ($newSlots as $key => $daysData) {
            if (!isset($oldSlots[$key]) || count($oldSlots[$key]) !== count($daysData)) {
                return true;
            }

            foreach ($daysData as $slotKey => $slotsData) {
                if (isset($oldSlots[$key][$slotKey])) {
                    $oldSlotData = $oldSlots[$key][$slotKey];
                    if (isset($oldSlotData['from']) && $oldSlotData['from'] !== $slotsData['from']) {
                        return true;
                    }

                    if (isset($oldSlotData['to']) && $oldSlotData['to'] !== $slotsData['to']) {
                        return true;
                    }

                    if (isset($oldSlotData['qty']) && $oldSlotData['qty'] !== $slotsData['qty']) {
                        return true;
                    }
                } else {
                    return true;
                }
            }
        }
    }

    /**
     * Save Hotel Booking Product.
     *
     * @param array $data
     * @param int $productId
     * @param int $productSetId
     * @param array $files
     * @return void
     */
    public function saveHotelBooking($data, $productId, $productSetId, $files)
    {
        try {
            $totalSlots = 0;
            $slotProductId = 0;
            $bookingProductId = 0;
            $helper = $this->helper;
            $infoModel = $this->info->create();
            $slotModel = $this->slot->create();

            if (empty($data['product_id']) && !empty($files) && !empty($files['amenities_icon'])) {
                foreach ($files['amenities_icon'] as $key => $amenitiyIcon) {
                    $target = $this->mediaDirectory->getAbsolutePath(
                        'catalog/product/' . $productId . '/' . $key . '/'
                    );
                    $removeDir = $this->filesystem->getDirectoryRead(
                        DirectoryList::MEDIA
                    )->getAbsolutePath(
                        'catalog/product/' . $productId . '/' . $key . '/'
                    );
                    $this->deleteImage($removeDir);
                    if (!empty($amenitiyIcon['tmp_name'])) {
                        $data = $this->uploadImageToDirectory($target, $key);
                    }
                }
            }

            if (empty($data['product']['configurable_attributes_data'])) {
                return false;
            }

            $bookingOptions = $helper->getJsonDecodedString($data['configurable-matrix-serialized']);

            if (empty($bookingOptions)) {
                return false;
            }
            $newBookingOptions = [];
            $hotelBookingProduct = $helper->getProduct($productId);
            $productTypeInstance = $hotelBookingProduct->getTypeInstance();
            $usedProducts = $productTypeInstance->getUsedProducts($hotelBookingProduct);
            foreach ($usedProducts as $child) {
                $childStockData = $helper->getStockData($child->getId());
                $newBookingOptions[$child->getId()] = [
                    'sku' => $child->getSku(),
                    'qty' => $childStockData->getQty(),
                    'price' => $child->getPrice(),
                    'status' => $child->getStatus(),
                ];
                $totalSlots += (int)$childStockData->getQty();
            }
            $qty = $totalSlots;
            $bookingOptionsData = $helper->getJsonEcodedString($newBookingOptions);

            $slotCollection = $slotModel->getCollection()
                ->addFieldToFilter("product_id", $productId);
            foreach ($slotCollection as $value) {
                $slotProductId = $value->getId();
            }
            $slotData = [
                'product_id' => $productId,
                'type' => 1,
                'status' => 1
            ];
            $slotModel->load($slotProductId)
                ->addData($slotData)
                ->save();

            $bookingCollection = $infoModel->getCollection()
                ->addFieldToFilter("product_id", $productId);
            foreach ($bookingCollection as $value) {
                $bookingProductId = $value->getId();
            }
            $infoData = [
                'attribute_set_id' => $productSetId,
                'product_id' => $productId,
                'type' => 1,
                'start_date' => '',
                'end_date' => '',
                'info' => $bookingOptionsData,
                'qty' => $qty,
                'total_slots' => $totalSlots
            ];
            $infoModel->load($bookingProductId)
                ->addData($infoData)
                ->save();
        } catch (\Exception $e) {
            $this->helper->logDataInLogger($e->getMessage());
        }
    }

    /**
     * Process source items for configurable matrix hotel booking products.
     *
     * @param array $hotelProduct
     * @param string $configurableMatrix
     *
     * @return void
     */
    public function processSourceItems($hotelProduct, $configurableMatrix)
    {
        if ($hotelProduct->getTypeId() !== 'hotelbooking') {
            return;
        }

        if ($configurableMatrix != '') {
            $hotelProductsData = json_decode($configurableMatrix, true);
            foreach ($hotelProductsData as $key => $hotelProductData) {
                if (isset($hotelProductsData['quantity_per_source'])) {
                    $quantityPerSource = is_array($hotelProductData['quantity_per_source'])
                        ? $hotelProductData['quantity_per_source']
                        : [];

                    // get sku by child id, because child sku can be changed if product with such sku already exists.
                    $childProductId = $hotelProduct->getExtensionAttributes()->getConfigurableProductLinks()[$key];
                    $childProductSku = $this->getSkusByProductIdsInterface->execute([$childProductId])[$childProductId];
                    try {
                        $this->processHotelSourceItems($quantityPerSource, $childProductSku);
                    } catch (\Magento\Framework\Exception\InputException $e) {
                        $this->helper->logDataInLogger($e->getLogMessage());
                        continue;
                    }
                }
            }
        }
    }

    /**
     * Process source items for given hotel booking product sku.
     *
     * @param array $hotelSourceItems
     * @param string $hotelProductSku
     * @return void
     * @throws \Magento\Framework\Exception\InputException
     */
    private function processHotelSourceItems(array $hotelSourceItems, string $hotelProductSku)
    {
        foreach ($hotelSourceItems as $key => $hotelSourceItem) {
            $hotelSourceItems[$key][SourceItemInterface::QUANTITY] = $hotelSourceItems[$key]['quantity_per_source'];

            if (!isset($hotelSourceItem[SourceItemInterface::STATUS])) {
                $hotelSourceItems[$key][SourceItemInterface::STATUS]
                    = $hotelSourceItems[$key][SourceItemInterface::QUANTITY] > 0 ? 1 : 0;
            }
        }

        $this->sourceItemsProcessor->execute($hotelProductSku, $hotelSourceItems);
    }

    /**
     * [deleteImage deletes image]
     *
     * @param  [string] $path [contains path]
     * @return [object|boolean]
     */
    public function deleteImage($path)
    {
        try {
            $directory = $this->mediaDirectory;
            $result = $directory->delete($directory->getRelativePath($path));
            return true;
        } catch (\Exception $e) {
            $this->helper->logDataInLogger("deleteImage : " . $e->getMessage());
            return false;
        }
    }

    /**
     * Upload Image To Directory
     *
     * @param  [string] $target [contains target path]
     * @param  string $fileId
     * @return void
     */
    public function uploadImageToDirectory($target, $fileId)
    {
        try {
            $uploader = $this->fileUploaderFactory
                ->create(
                    ['fileId' => 'amenities_icon[' . $fileId . ']']
                );
            $image = $uploader->validateFile();

            if (isset($image['tmp_name'])
                && $image['tmp_name'] !== ''
                && $image['tmp_name'] !== null
            ) {
                //$imageCheck = getimagesize($image['tmp_name']);
                $imageData = $this->filesystemDriverFile->fileGetContents($image['tmp_name']);
                $imageCheck = getimagesizefromstring($imageData);

                if ($imageCheck['mime']) {
                    $image['name'] = str_replace(" ", "_", $image['name']);
                    $imgName = rand(1, 99999) . $image['name'];

                    $uploader->setAllowedExtensions(
                        ['jpg', 'jpeg', 'gif', 'png']
                    );
                    $uploader->setAllowRenameFiles(true);
                    $result = $uploader->save($target, $imgName);
                }
            }
        } catch (\Exception $e) {
            $this->helper->logDataInLogger("uploadImageToDirectory : " . $e->getMessage());
        }
    }

    /**
     * Save Table Booking Product.
     *
     * @param array $data
     * @param int $productId
     * @param int $productSetId
     * @return void
     */
    public function saveTableBooking($data, $productId, $productSetId)
    {
        try {
            $helper = $this->helper;
            $infoModel = $this->info->create();
            $slotModel = $this->slot->create();
            if (empty($data['product']['slot_data'])) {
                return false;
            }
            $data['product'] = $this->getCalculatedSlotDataForTableBooking($data['product']);
            $totalSlots = $data['product']['total_slots'];
            $qty = 0;

            if (isset($data['product']['quantity_and_stock_status']['qty'])) {
                $qty = $data['product']['quantity_and_stock_status']['qty'];
            } elseif (isset($data['product']['stock_data']['qty'])) {
                $qty = $data['product']['stock_data']['qty'];
            }

            if ($data['product']['slot_for_all_days']) {
                $data['product']['slot_data'][2] = $data['product']['slot_data'][1];
                $data['product']['slot_data'][3] = $data['product']['slot_data'][1];
                $data['product']['slot_data'][4] = $data['product']['slot_data'][1];
                $data['product']['slot_data'][5] = $data['product']['slot_data'][1];
                $data['product']['slot_data'][6] = $data['product']['slot_data'][1];
                $data['product']['slot_data'][7] = $data['product']['slot_data'][1];
            }

            $slotDuration = $data['product']['slot_duration'];
            $slotBreakTime = $data['product']['break_time_bw_slot'];
            $totalSlotTime = $slotDuration + $slotBreakTime;
            foreach ($data['product']['slot_data'] as $dayKey => $dayValue) {
                $slotIndex = 1;
                $currentSlots = 0;
                if (!empty($dayValue)) {
                    foreach ($dayValue as $key => $value) {
                        if (empty($value['qty'])) {
                            $value['qty'] = $data['product']['max_capacity'];
                        }
                        $availableSlotArr = [];
                        $slotFromTimeStamp = strtotime($value['from']);
                        $slotToTimeStamp = strtotime($value['to']);
                        $totalTimeStamp = ($slotToTimeStamp - $slotFromTimeStamp) / 60; // in hours
                        $currentSlots = $currentSlots + (int)($totalTimeStamp / $totalSlotTime);
                        $availableSlotArr[$slotIndex - 1]['qty'] = $value['qty'];
                        $availableSlotArr[$slotIndex - 1]['time'] = $value['from'];
                        for ($i = $slotIndex; $i < $currentSlots; $i++) {
                            $availableSlotArr[$i]['qty'] = $value['qty'];
                            $availableSlotArr[$i]['time'] = date(
                                "h:i a",
                                strtotime(
                                    '+' . $totalSlotTime . ' minutes',
                                    strtotime($availableSlotArr[$i - 1]['time'])
                                )
                            );
                        }
                        $slotIndex = $currentSlots;
                        $slotIndex++;
                        $data['product']['slot_data'][$dayKey][$key]['slots_info'] = $availableSlotArr;
                    }
                }
            }
            $bookingSlotData = $helper->getJsonEcodedString($data['product']['slot_data']);

            $slotProductId = 0;
            $slotCollection = $slotModel->getCollection()
                ->addFieldToFilter("product_id", $productId);
            foreach ($slotCollection as $value) {
                $slotProductId = $value->getId();
            }

            $infoModelOld = null;
            $bookingProductId = 0;
            $bookingCollection = $infoModel->getCollection()
                ->addFieldToFilter("product_id", $productId);
            foreach ($bookingCollection as $value) {
                $bookingProductId = $value->getId();
                $infoModelOld = $value;
            }
            $type = $data['product']['slot_for_all_days'];
            $slotQty = $data['product']['max_capacity'];
            if ($this->canSaveTableSlots($infoModelOld, $bookingSlotData, $type, $slotQty)) {
                $infoData = [
                    'attribute_set_id' => $productSetId,
                    'product_id' => $productId,
                    'type' => $type,
                    'info' => $bookingSlotData,
                    'qty' => $qty,
                    'total_slots' => $totalSlots
                ];
                $slotData = [
                    'product_id' => $productId,
                    'type' => $type,
                    'status' => 1
                ];
                $slotModel->load($slotProductId)
                    ->setData($slotData)
                    ->save();

                $infoModel->load($bookingProductId)
                    ->addData($infoData)
                    ->save();
            }
        } catch (\Exception $e) {
            $this->helper->logDataInLogger($e->getMessage());
        }
    }

    /**
     * Can Table Booking Product for request Slots Method.
     *
     * @param object $infoModelOld
     * @param string $newJsonData
     * @param string $type
     * @param int $slotQty
     * @return bool
     */
    public function canSaveTableSlots($infoModelOld, $newJsonData, $type, $slotQty)
    {
        if (empty($infoModelOld) || $infoModelOld->getType() != $type) {
            return true;
        }

        $oldJson = $this->helper->getJsonDecodedString($infoModelOld->getInfo());
        $newJson = $this->helper->getJsonDecodedString($newJsonData);

        if ($this->checkTableSlotsData($oldJson, $newJson, $slotQty)) {
            return true;
        }
    }

    /**
     * Check Table Booking Product for request Slots Method.
     *
     * @param array $oldSlots
     * @param array $newSlots
     * @param int $slotQty
     *
     * @return bool
     */
    public function checkTableSlotsData($oldSlots, $newSlots, $slotQty)
    {
        if (count($oldSlots) !== count($newSlots)) {
            return true;
        }
        foreach ($newSlots as $key => $daysData) {
            if (!isset($oldSlots[$key]) || count($oldSlots[$key]) !== count($daysData)) {
                return true;
            }

            foreach ($daysData as $slotKey => $slotsData) {
                if (!isset($oldSlots[$key][$slotKey])) {
                    return true;
                }
                $oldSlotData = $oldSlots[$key][$slotKey];
                if (isset($oldSlotData['from']) && $oldSlotData['from'] !== $slotsData['from']) {
                    return true;
                }

                if (isset($oldSlotData['to']) && $oldSlotData['to'] !== $slotsData['to']) {
                    return true;
                }

                if (isset($slotsData['slots_info']) && isset($oldSlotData['slots_info'])) {
                    $oldSlotInfo = $oldSlotData['slots_info'];
                    foreach ($slotsData['slots_info'] as $slotInfoKey => $slotInfo) {
                        if (isset($oldSlotInfo[$slotInfoKey]['qty'])
                            && $oldSlotInfo[$slotInfoKey]['qty'] !== $slotInfo['qty']
                        ) {
                            return true;
                        }

                        if (isset($oldSlotInfo[$slotInfoKey]['time'])
                            && $oldSlotInfo[$slotInfoKey]['time'] !== $slotInfo['time']
                        ) {
                            return true;
                        }
                    }
                } else {
                    return true;
                }
            }
        }
    }

    /**
     * Save Default Booking Product.
     *
     * @param array $data
     * @param string $productId
     * @return void
     */
    public function saveDefaultBooking($data, $productId)
    {
        try {
            $helper = $this->helper;
            $infoModel = $this->info->create();
            $slotModel = $this->slot->create();
            $bookingType = $this->getBookingType($data, $productId);
            $isNew = false;
            $collection = $infoModel->getCollection()
                ->addFieldToFilter("product_id", $productId);
            if ($collection->getSize() <= 0) {
                $isNew = true;
            }
            if ($collection->getSize() <= 0 && $bookingType == 0) {
                return;
            } else {
                $previousBookingType = $helper->getBookingType($productId);
                if ($bookingType == 0 && $previousBookingType == 0) {
                    return;
                }
            }
            if ($bookingType == 0 || ($bookingType == 2 && !array_key_exists("start", $data['info']))) {
                $helper->disableSlots($productId);
                $helper->deleteInfo($productId);
                return;
            }

            if (!array_key_exists("info", $data)) {
                return;
            }

            $startDate = $data['start_date'];
            $endDate = $data['end_date'];
            $qty = 0;

            if (isset($data['product']['quantity_and_stock_status']['qty'])) {
                $qty = $data['product']['quantity_and_stock_status']['qty'];
            } elseif (isset($data['product']['stock_data']['qty'])) {
                $qty = $data['product']['stock_data']['qty'];
            }

            $result = $this->prepareOptions($data, $bookingType);

            if (!empty($result)) {
                $bookingInfo = $result['info'];

                $count = $result['total'];
                //Setting Booking Information
                $bookingInfo = $helper->getJsonEcodedString($bookingInfo);
                if (!$isNew && $isNew == false) {
                    $bookingData = $helper->getBookingInfo($productId);

                    if ($this->canSaveSlots($bookingData, $bookingInfo, $qty)) {
                        $helper->disableSlots($productId);
                        $slotData = [
                            'product_id' => $productId,
                            'type' => $bookingType,
                            'status' => 1
                        ];

                        $slotModel->setData($slotData)->save();
                    }
                } else {
                    $slotData = [
                        'product_id' => $productId,
                        'type' => $bookingType,
                        'status' => 1
                    ];
                    $slotModel->setData($slotData)->save();
                }

                $infoData = [
                    'product_id' => $productId,
                    'type' => $bookingType,
                    'start_date' => $startDate,
                    'end_date' => $endDate,
                    'info' => $bookingInfo,
                    'qty' => $qty,
                    'total_slots' => $count
                ];
                $collection = $infoModel->getCollection();
                $item = $helper->getDataByField($productId, 'product_id', $collection);
                if ($item) {
                    $id = $item->getId();
                    $infoModel->addData($infoData)->setId($id)->save();
                } else {
                    $infoModel->setData($infoData)->save();
                }
                $product = $helper->getProduct($productId);
                $productExtension = $product->getExtensionAttributes();
                $manageStock = $productExtension->getStockItem()->getManageStock();
                if ($manageStock) {
                    $productExtension->getStockItem()->setUseConfigManageStock(0);
                    $productExtension->getStockItem()->setManageStock(0);
                }
            }
        } catch (\Exception $e) {
            $this->helper->logDataInLogger($e->getMessage());
        }
    }

    /**
     * Get Booking Type
     *
     * @param array $data
     * @param int $productId
     *
     * @return int
     */
    public function getBookingType($data, $productId)
    {
        $bookingType = 0;
        try {
            if (array_key_exists("booking_type", $data)) {
                $bookingType = $data['booking_type'];
            }
        } catch (\Exception $e) {
            $this->helper->logDataInLogger(
                "Observer_AfterProductSave getBookingType : " . $e->getMessage()
            );
        }

        return (int)$bookingType;
    }

    /**
     * Check Whether New Slots or Not
     *
     * @param array  $bookingData
     * @param string $bookingInfo
     * @param int    $qty
     *
     * @return bool
     */
    public function canSaveSlots($bookingData, $bookingInfo, $qty)
    {
        try {
            if ($bookingData['is_booking']) {
                if (is_array($bookingData['info'])) {
                    $tempInfo = $this->helper->getJsonEcodedString($bookingData['info']);
                } else {
                    $tempInfo = $bookingData['info'];
                }

                if (strcmp($bookingInfo, $tempInfo) !== 0) {
                    return true;
                } else {
                    if ($bookingData['qty'] != $qty) {
                        return true;
                    }
                }
            } else {
                return true;
            }
        } catch (\Exception $e) {
            $this->helper->logDataInLogger("Observer_AfterProductSave_canSaveSlots Exception : " . $e->getMessage());
        }

        return false;
    }

    /**
     * Prepare Options
     *
     * @param array $data
     * @param int   $bookingType
     *
     * @return array
     */
    public function prepareOptions($data, $bookingType)
    {
        $result = [];
        try {
            if ($bookingType == 1) {
                $result = $this->prepareManyBookingOptions($data);
            } elseif ($bookingType == 2) {
                $result = $this->prepareOneBookingOptions($data);
            }
        } catch (\Exception $e) {
            $this->helper->logDataInLogger("Observer_AfterProductSave prepareOptions : " . $e->getMessage());
        }
        return $result;
    }

    /**
     * Prepare Many Booking Options
     *
     * @param array $data
     *
     * @return array
     */
    public function prepareManyBookingOptions($data)
    {
        try {
            $count = 1;
            $info = $data['info'];
            $startDate = $data['start_date'];
            $endDate = $data['end_date'];
            $slotTime = $data['time_slot'];
            $breakTime = $data['break_time'];
            $numOfDays = $this->getDateDifference($startDate, $endDate);
            for ($i = 0; $i <= $numOfDays; $i++) {
                $date = strtotime("+$i day", strtotime($startDate));
                $day = strtolower(date("l", $date));
                $status = $info[$day]['status'];
                if ($status == 1) {
                    $startHour = $info[$day]['start_hour'];
                    $startMinute = $info[$day]['start_minute'];
                    $endHour = $info[$day]['end_hour'];
                    $endMinute = $info[$day]['end_minute'];
                    $startCount = $startHour * 60 + $startMinute;
                    $endCount = $endHour * 60 + $endMinute;
                    $diff = $endCount - $startCount;
                    while ($diff >= $slotTime) {
                        $diff = $diff - ($breakTime + $slotTime);
                        $count++;
                    }
                }
            }

            unset($data['info']['start_hour']);
            unset($data['info']['start_minute']);
            unset($data['info']['end_hour']);
            unset($data['info']['end_minute']);
            $bookingInfo = $data['info'];
            $bookingInfo['time_slot'] = $slotTime;
            $bookingInfo['break_time'] = $breakTime;
            $result = [];
            $result['info'] = $bookingInfo;
            $result['start_date'] = $startDate;
            $result['end_date'] = $endDate;
            $result['total'] = $count - 1;
            return $result;
        } catch (\Exception $e) {
            $this->helper->logDataInLogger("Observer_AfterProductSave prepareManyBookingOptions : " . $e->getMessage());
        }
    }

    /**
     * Prepare One Booking Options
     *
     * @param array $data
     *
     * @return array
     */
    public function prepareOneBookingOptions($data)
    {
        try {
            $count = 1;
            $startDate = $data['start_date'];
            $endDate = $data['end_date'];
            $numOfDays = $this->getDateDifference($startDate, $endDate);
            $startData = $data['info']['start'];
            $endData = $data['info']['end'];
            $startDays = $startData['day'];

            for ($i = 0; $i <= $numOfDays; $i++) {
                $date = strtotime("+$i day", strtotime($startDate));
                $day = strtolower(date("l", $date));
                $date = strtolower(date("Y-m-d", $date));
                foreach ($startDays as $key => $startDay) {
                    if ($day == $startDay) {
                        $count++;
                    }
                }
            }

            $bookingInfo = ['start' => $startData, 'end' => $endData];
            $result = [];
            $result['info'] = $bookingInfo;
            $result['total'] = $count - 1;
            return $result;
        } catch (\Exception $e) {
            $this->helper->logDataInLogger(
                "Observer_AfterProductSave_prepareOneBookingOptions Exception : " . $e->getMessage()
            );
        }
    }

    /**
     * Get Difference  of Dates
     *
     * @param string $firstDate
     * @param string $lastDate
     *
     * @return int
     */
    public function getDateDifference($firstDate, $lastDate)
    {
        try {
            $date1 = date_create($firstDate);
            $date2 = date_create($lastDate);
            $diff = date_diff($date1, $date2);
            $numOfDays = (int)$diff->format("%R%a");
            return $numOfDays;
        } catch (\Exception $e) {
            $this->helper->logDataInLogger("Observer_AfterProductSave getDateDifference : " . $e->getMessage());
        }
    }

    /**
     * Get Slot Data
     *
     * @param array $data
     *
     * @return array $data
     */
    public function getCalculatedSlotData($data)
    {
        try {
            $count = 1;
            $startDate = $data['booking_available_from'];
            $endDate = $data['booking_available_to'];
            $days = 7;
            $totalDaysCountPerWeek = 0;
            if ($data['slot_for_all_days']) {
                $days = 1;
                $totalDaysCountPerWeek = 7;
            }
            $totalSlots = 0;
            $dayWiseSlots = [];
            for ($i = 1; $i <= $days; $i++) {
                if (empty($data['slot_data'][$i])) {
                    continue;
                }

                if (!$data['slot_for_all_days']) {
                    $totalDaysCountPerWeek++;
                }
                //sorting slots accoding to from time
                usort($data['slot_data'][$i], function ($a, $b) {
                    return (int)(strtotime($a['from']) > strtotime($b['from']));
                });
                $previousTo = -1;
                foreach ($data['slot_data'][$i] as $key => $value) {
                    $slotFromTimeStamp = strtotime($value['from']);
                    $slotToTimeStamp = strtotime($value['to']);

                    //checking slot duplicacy
                    if ($slotFromTimeStamp >= $previousTo) {
                        $previousTo = $slotToTimeStamp;
                    } else {
                        unset($data['slot_data'][$i][$key]);
                        continue;
                    }

                    if ($slotToTimeStamp > $slotFromTimeStamp) {
                        $totalTimeStamp = ($slotToTimeStamp - $slotFromTimeStamp) / 60; // in hours
                    } else {
                        unset($data['slot_data'][$i][$key]);
                        $totalTimeStamp = 0;
                    }
                    if (empty($totalTimeStamp)) {
                        continue;
                    }

                    $slotDuration = (float)$data['slot_duration'];
                    $slotBreakTime = (float)$data['break_time_bw_slot'];
                    $currentSlots = (int)(($totalTimeStamp) / ($slotDuration + $slotBreakTime));
                    if (!empty($data['slot_data'][$i][$key]['qty'])) {
                        $currentSlots = $currentSlots * $data['slot_data'][$i][$key]['qty'];
                    } else {
                        $data['slot_data'][$i][$key]['qty'] = 1;
                    }
                    $totalSlots = $totalSlots + $currentSlots;
                    if (isset($dayWiseSlots[$i])) {
                        $dayWiseSlots[$i] = $dayWiseSlots[$i] + $currentSlots;
                    } else {
                        $dayWiseSlots[$i] = $currentSlots;
                    }
                }
            }

            if ($data['available_every_week']) {
                $data['total_slots'] = 99999999;
                return $data;
            }

            $numOfDays = $this->getDateDifference($startDate, $endDate);
            $daysCount = [];
            if (!$data['slot_for_all_days']) {
                if ($numOfDays > 0) {
                    $from = date_create($startDate);
                    $to = date_create($endDate);
                    $to->modify('+1 day');
                    $interval = new \DateInterval('P1D');
                    $periods = new \DatePeriod($from, $interval, $to);

                    foreach ($periods as $period) {
                        if (isset($daysCount[$period->format('w')])) {
                            $daysCount[$period->format('w')] = $daysCount[$period->format('w')] + 1;
                        } else {
                            $daysCount[$period->format('w')] = 1;
                        }
                    }
                }
            } else {
                if (!empty($dayWiseSlots)) {
                    $allSlots = 0;
                    foreach ($dayWiseSlots as $slots) {
                        $allSlots = $allSlots + $slots;
                    }
                    $totalSlots = ($numOfDays + 1) * $allSlots;
                }
            }

            if (!empty($daysCount)) {
                $totalSlots = 0;
                foreach ($daysCount as $dayNumber => $count) {
                    $dayIndex = ($dayNumber == 0) ? 7 : $dayNumber;

                    if (isset($dayWiseSlots[$dayIndex])) {
                        $totalSlots = $totalSlots + ($dayWiseSlots[$dayIndex] * $count);
                    }
                }
            }
        } catch (\Exception $e) {
            $totalSlots = 0;
            $this->helper->logDataInLogger($e->getMessage());
        }
        $data['total_slots'] = $totalSlots;
        return $data;
    }

    /**
     * Get Rent Slot Data
     *
     * @param array $data
     *
     * @return array $data
     */
    public function getCalculatedRentSlotData($data)
    {
        try {
            $this->helper->logDataInLogger(json_encode($data));
            $count = 1;
            $startDate = $data['booking_available_from'];
            $endDate = $data['booking_available_to'];
            $days = 7;
            $totalDaysCountPerWeek = 0;
            if ($data['slot_for_all_days']) {
                $days = 1;
                $totalDaysCountPerWeek = 7;
            }
            $totalSlots = 0;
            for ($i = 1; $i <= $days; $i++) {
                if (!empty($data['slot_data'][$i])) {
                    if (!$data['slot_for_all_days']) {
                        $totalDaysCountPerWeek++;
                    }
                    //sorting slots accoding to from time
                    usort($data['slot_data'][$i], function ($a, $b) {
                        return (int)(strtotime($a['from']) > strtotime($b['from']));
                    });
                    $previousTo = -1;
                    foreach ($data['slot_data'][$i] as $key => $value) {
                        $slotFromTimeStamp = strtotime($value['from']);
                        $slotToTimeStamp = strtotime($value['to']);

                        //checking slot duplicacy and time validity
                        if ($slotFromTimeStamp >= $previousTo && $slotToTimeStamp > $slotFromTimeStamp) {
                            $previousTo = $slotToTimeStamp;
                        } else {
                            unset($data['slot_data'][$i][$key]);
                            continue;
                        }

                        if (empty($data['slot_data'][$i][$key]['qty'])) {
                            $data['slot_data'][$i][$key]['qty'] = 1;
                        }
                        $totalSlots = $totalSlots + $data['slot_data'][$i][$key]['qty'];
                    }
                }
            }
            if ($data['available_every_week']) {
                $totalSlots = 99999999;
            } else {
                $numOfDays = $this->getDateDifference($startDate, $endDate);
                $totalWeeks = round($numOfDays / 7);
                if ($totalWeeks) {
                    $totalDays = $totalWeeks * $totalDaysCountPerWeek;
                    $totalSlots = $totalSlots * $totalDays;
                }
            }
        } catch (\Exception $e) {
            $totalSlots = 0;
            $this->helper->logDataInLogger($e->getMessage());
        }
        $data['total_slots'] = $totalSlots;
        return $data;
    }

    /**
     * Check Hotel Child Product
     *
     * @param int $productId
     * @param array $data
     */
    public function checkHotelChildProduct($productId, $data)
    {
        $parentProducts = $this->configurable->getParentIdsByChild($productId);
        if ($parentProducts && !empty($parentProducts) && is_array($parentProducts)) {
            $collection = $this->infoCollection->create()
                ->addFieldToFilter("product_id", ['in' => $parentProducts]);
            if ($collection->getSize()) {
                foreach ($collection as $info) {
                    $bookingInfoData = $this->helper->getJsonDecodedString(
                        $info->getInfo()
                    );
                    if (array_key_exists($productId, $bookingInfoData)) {
                        $qty = 0;

                        if (isset($data['product']['quantity_and_stock_status']['qty'])) {
                            $qty = $data['product']['quantity_and_stock_status']['qty'];
                        } elseif (isset($data['product']['stock_data']['qty'])) {
                            $qty = $data['product']['stock_data']['qty'];
                        }

                        $updatedQty = $qty;
                        $bookingInfoData[$productId]['qty'] = $updatedQty;
                        $infoData = $this->helper->getJsonEcodedString($bookingInfoData);
                        $info->setInfo($infoData)->save();
                    }
                }
            }
        }
    }

    /**
     * Get Slot Data for Table type Booking
     *
     * @param array $data
     *
     * @return array $data
     */
    public function getCalculatedSlotDataForTableBooking($data)
    {
        try {
            $count = 1;
            $days = 7;
            $totalSlots = 0;
            $totalDaysCountPerWeek = 0;
            if ($data['slot_for_all_days']) {
                $days = 1;
                $totalDaysCountPerWeek = 7;
            }
            for ($i = 1; $i <= $days; $i++) {
                if (!empty($data['slot_data'][$i])) {
                    if (!$data['slot_for_all_days']) {
                        $totalDaysCountPerWeek++;
                    }
                    //sorting slots accoding to from time
                    usort($data['slot_data'][$i], function ($a, $b) {
                        return (int)(strtotime($a['from']) > strtotime($b['from']));
                    });
                    $previousTo = -1;
                    foreach ($data['slot_data'][$i] as $key => $value) {
                        $slotFromTimeStamp = strtotime($value['from']);
                        $slotToTimeStamp = strtotime($value['to']);

                        //checking slot duplicacy
                        if ($slotFromTimeStamp >= $previousTo) {
                            $previousTo = $slotToTimeStamp;
                        } else {
                            unset($data['slot_data'][$i][$key]);
                            continue;
                        }

                        if ($slotToTimeStamp > $slotFromTimeStamp) {
                            $totalTimeStamp = ($slotToTimeStamp - $slotFromTimeStamp) / 60; // in hours
                        } else {
                            unset($data['slot_data'][$i][$key]);
                            $totalTimeStamp = 0;
                        }
                        if ($totalTimeStamp) {
                            $slotDuration = (float)$data['slot_duration'];
                            $slotBreakTime = (float)$data['break_time_bw_slot'];
                            $currentSlots = (int)(($totalTimeStamp) / ($slotDuration + $slotBreakTime));
                            $totalSlots += $currentSlots;
                        }
                    }
                }
            }
            $totalSlots = 99999999;
        } catch (\Exception $e) {
            $totalSlots = 0;
            $this->helper->logDataInLogger($e->getMessage());
        }
        $data['total_slots'] = $totalSlots;
        return $data;
    }
}
