<?php
/**
 * Webkul Software.
 *
 * @category  Webkul
 * @package   Webkul_AdvancedBookingSystem
 * @author    Webkul Software Private Limited
 * @copyright Webkul Software Private Limited (https://webkul.com)
 * @license   https://store.webkul.com/license.html
 */
namespace Webkul\AdvancedBookingSystem\Observer;

use Magento\Framework\Event\ObserverInterface;
use Magento\Checkout\Model\Session as CheckoutSession;
use Magento\Framework\App\RequestInterface;
use Magento\Framework\Message\ManagerInterface;
use Webkul\AdvancedBookingSystem\Model\ResourceModel\Quote\CollectionFactory as QuoteCollection;
use Webkul\AdvancedBookingSystem\Model\Info;

class CheckoutCartProductAddAfterObserver implements ObserverInterface
{
    /**
     * @var CheckoutSession
     */
    private $checkoutSession;

    /**
     * @var RequestInterface
     */
    private $request;

    /**
     * @var \Webkul\AdvancedBookingSystem\Helper\Data
     */
    private $helper;

    /**
     * @var QuoteCollection
     */
    private $quoteCollection;

    /**
     * Constructor
     *
     * @param CheckoutSession                           $checkoutSession
     * @param RequestInterface                          $request
     * @param \Webkul\AdvancedBookingSystem\Helper\Data $helper
     * @param QuoteCollection                           $quoteCollectionFactory
     */
    public function __construct(
        CheckoutSession $checkoutSession,
        RequestInterface $request,
        \Webkul\AdvancedBookingSystem\Helper\Data $helper,
        QuoteCollection $quoteCollectionFactory
    ) {
        $this->checkoutSession = $checkoutSession;
        $this->request = $request;
        $this->helper = $helper;
        $this->quoteCollection = $quoteCollectionFactory;
    }

    /**
     * Checkout cart product add event handler.
     *
     * @param \Magento\Framework\Event\Observer $observer
     */
    public function execute(\Magento\Framework\Event\Observer $observer)
    {
        $this->helper->logDataInLogger('CheckoutCartProductAddAfterObserver ');
        $data = $this->request->getParams();
        $helper = $this->helper;
        $product = $observer->getEvent()->getProduct();
        $item = $observer->getEvent()->getQuoteItem();
        $quoteId = $item->getQuoteId();
        $productType = $product->getTypeId();
        if ($productType == 'booking' && !$item->getId()) {
            $productSetId = $product->getAttributeSetId();
            $allowedAttrSetIDs = $helper->getAllowedAttrSetIDs();
            $rentalAttrSetId = $helper->getProductAttributeSetIdByLabel(
                'Rental Booking'
            );
            if (in_array($productSetId, $allowedAttrSetIDs)) {
                if ($productSetId==$rentalAttrSetId) {
                    $this->rentTypeBookingAction($data, $quoteId, $product, $item);
                }
            }
        }
        if ($productType=="hotelbooking" && $item->getId()) {
            $collection = $this->quoteCollection->create();
            $bookingQuote = $helper->getDataByField($item->getId(), 'item_id', $collection);
            if ($bookingQuote && $bookingQuote->getItemId()==$item->getId()) {
                $helper->checkItemQtyAvilableForHotel($data, $product, $item, $bookingQuote);
            }
        }

        if ($productType == 'booking' && $item->getId()) {
            $productSetId = $product->getAttributeSetId();
            $allowedAttrSetIDs = $helper->getAllowedAttrSetIDs();
            $tableAttrSetId = $helper->getProductAttributeSetIdByLabel(
                'Table Booking'
            );

            if (in_array($productSetId, $allowedAttrSetIDs)) {
                if ($productSetId == $tableAttrSetId) {
                    $this->tableTypeBookingAction($data, $quoteId, $product, $item);
                }
            }
        }
    }

    /**
     * Rent Type Booking Action
     *
     * @param array $data
     * @param int $quoteId
     * @param object $product
     * @param object $item
     */
    public function rentTypeBookingAction($data, $quoteId, $product, $item)
    {
        $helper = $this->helper;
        $productId = $product->getId();
        $rentOpt = $helper->getRentOptions($product);
        if (!empty($rentOpt['choose_rent_type']['option_id']) && !empty($data['booking_date_from'])) {
            $rentType = '';
            $optionId = $rentOpt['choose_rent_type']['option_id'];
            $optionValues = $rentOpt['choose_rent_type']['option_values'];
            if (!empty($data['options'][$optionId]) && !empty($optionValues)) {
                $optionValId = $data['options'][$optionId];
                foreach ($optionValues as $key => $value) {
                    if ($optionValId == $value['option_type_id']) {
                        if ($value['title'] == 'Hourly Basis') {
                            $rentType = Info::RENT_TYPE_HOURLY;
                        } else {
                            $rentType = Info::RENT_TYPE_DAILY;
                        }
                        break;
                    }
                }
            }
            $bookedSlotFromDate = $data['booking_date_from'];
            if ($rentType == Info::RENT_TYPE_HOURLY) {
                $bookedSlotToDate = $data['booking_date_from'];
                //checking if there is a holiday on selected booking dates
                $holidayDate = date('Y-m-d', strtotime($bookedSlotFromDate));
                $holiday = $helper->checkHolidayForBooking($productId, $holidayDate);
                if ($holiday['status']) {
                    $errorMessage = __(
                        'Booking dates are invalid, booking is closed on %1 due to %2.',
                        $holiday['date'],
                        $holiday['name']
                    );
                    $this->checkoutSession->getQuote()->setHasError(true);
                    throw new \Magento\Framework\Exception\LocalizedException(
                        $errorMessage
                    );
                }
                $selectedBookingFromTime = '';
                $selectedBookingToTime = '';
            } else {
                $bookedSlotToDate = $data['booking_date_to'];
                $selectedBookingFromTime = date(
                    "h:i a",
                    strtotime($bookedSlotFromDate)
                );
                $selectedBookingToTime = date(
                    "h:i a",
                    strtotime($bookedSlotToDate)
                );
            }
            $selectedBookingFromDate = date(
                "Y-m-d",
                strtotime($bookedSlotFromDate)
            );
            $selectedBookingToDate = date(
                "Y-m-d",
                strtotime($bookedSlotToDate)
            );

            if ($rentType == Info::RENT_TYPE_HOURLY) {
                if (empty($data['slot_day_index'])) {
                    $data['parent_slot_id'] = 0;
                    $data['slot_id'] = 0;
                    $data['slot_day_index'] = 0;
                    $data['booking_from_time'] = 0;
                    $data['booking_to_time'] = 0;
                }
                $parentSlotId = $data['parent_slot_id'];
                $slotId = $data['slot_id'];
                $slotDayIndex = $data['slot_day_index'];
                $slotIdFrom = $data['booking_from_time'];
                $slotIdTo = $data['booking_to_time'];

                $bookingInfo = $helper->getBookingInfo($productId);
                $bookingSlotData = $helper->getJsonDecodedString(
                    $bookingInfo['info']
                );

                $isSlotExisted = 0;
                if (!empty($bookingSlotData[$slotDayIndex][$parentSlotId]['slots_info'][$slotIdFrom])
                    && !empty($bookingSlotData[$slotDayIndex][$parentSlotId]['slots_info'][$slotIdTo])
                ) {
                    $isSlotExisted = 1;
                }
                if ($data['slot_day_index'] && $isSlotExisted) {
                    $bookingSlotDataArr = $bookingSlotData[$slotDayIndex][$parentSlotId];
                    $slotDataFrom = $bookingSlotDataArr['slots_info'][$slotIdFrom];
                    $slotDataTo = $bookingSlotDataArr['slots_info'][$slotIdTo];
                    if (!empty($slotDataFrom['time']) && !empty($slotDataTo['time'])) {
                        $selectedBookingFromTime = $slotDataFrom['time'];
                        $selectedBookingToTime = $slotDataTo['time'];
                    }
                    $rentPeriodArr = $this->checkoutSession->getRentPeriod();
                    // number of hours for rent
                    $hourDiff = strtotime($selectedBookingToTime) - strtotime($selectedBookingFromTime);
                    $rentPeriod = round($hourDiff/(60*60));
                    if (!$rentPeriod) {
                        $rentPeriod = 1;
                    }
                    // update item price
                    $price = $helper->getCovertedPrice($item->getProduct()->getFinalPrice());
                    $item->setCustomPrice($price*$rentPeriod);
                    $item->setOriginalCustomPrice($price*$rentPeriod);
                }
            } elseif ($rentType == Info::RENT_TYPE_DAILY) {
                $rentPeriodArr = $this->checkoutSession->getRentPeriod();
                // number of days for rent
                $dateDiff = strtotime($data['booking_date_to']) - strtotime($data['booking_date_from']);
                $rentPeriod = round($dateDiff/(60*60*24));
                if (!$rentPeriod) {
                    $rentPeriod = 1;
                }
                if (strtotime($data['booking_date_to']) != strtotime($data['booking_date_from'])) {
                    $rentPeriod++;
                }
                // update item price
                $price = $helper->getCovertedPrice($item->getProduct()->getFinalPrice());
                $item->setCustomPrice($price*$rentPeriod);
                $item->setOriginalCustomPrice($price*$rentPeriod);
            }
        }
    }

    /**
     * Table Type Booking Action
     *
     * @param array $data
     * @param int $quoteId
     * @param object $product
     * @param object $item
     */
    public function tableTypeBookingAction($data, $quoteId, $product, $item)
    {
        $helper = $this->helper;
        $productId = $product->getId();
        $collection = $this->quoteCollection->create();
        $bookingQuote = $helper->getDataByField($item->getId(), 'item_id', $collection);
        if ($bookingQuote && $bookingQuote->getItemId()==$item->getId()) {
            $helper->checkItemQtyAvilableForTable($data, $product, $item, $bookingQuote);
            $this->checkRequestedQtyAvailableForTable($data, $product, $item, $bookingQuote);
        }
    }

    /**
     * Check Requested Quantity for table
     *
     * @param array $data
     * @param object $product
     * @param object $item
     * @param object $bookingQuote
     */
    public function checkRequestedQtyAvailableForTable($data, $product, $item, $bookingQuote)
    {
        if (!empty($data['booking_date']) && !empty($data['booking_time'])) {
            $helper = $this->helper;
            $selectedBookingDate = $data['booking_date'];
            $selectedBookingTime = trim($data['booking_time']);
            $bookedSlotDate = date(
                "d M, Y",
                strtotime($selectedBookingDate)
            )." ".trim($selectedBookingTime);
            $availableQty = 0;
            $productId = $product->getId();
            if (empty($data['slot_day_index'])) {
                $data['slot_day_index'] = 0;
            }
            if (empty($data['slot_id'])) {
                $data['slot_id'] = 0;
            }
            if (empty($data['parent_slot_id'])) {
                $data['parent_slot_id'] = 0;
            }
            $parentSlotId = $data['parent_slot_id'];
            $slotId = $data['slot_id'];
            $slotDayIndex = $data['slot_day_index'];

            $bookedData = $helper->getBookedAppointmentDates($productId);
            
            $bookingInfo = $helper->getBookingInfo($productId);
            $bookingSlotData = $helper->getJsonDecodedString(
                $bookingInfo['info']
            );
            if (!empty($bookingSlotData[$slotDayIndex][$parentSlotId]['slots_info'][$slotId])) {
                $slotData = $bookingSlotData[$slotDayIndex][$parentSlotId]['slots_info'][$slotId];
                if (!empty($slotData['qty'])) {
                    $availableQty = $slotData['qty'];
                }
            }
            if (!empty($bookedData[strtotime($selectedBookingDate)][$selectedBookingTime])) {
                $bookedQty = $bookedData[strtotime($selectedBookingDate)][$selectedBookingTime];
                if ($bookedQty > $availableQty) {
                    $availableQty = 0;
                } else {
                    $availableQty = $availableQty - $bookedQty;
                }
            }
            $requestedQty = $item->getQty();
            if (!empty($data['charged_per_count']) && $data['charged_per_count'] > 1) {
                $requestedQty = $requestedQty * $data['charged_per_count'];
            }
            if (!$availableQty) {
                $errorMessage = __(
                    '%1 quantity is not available for slot %2.',
                    $item->getName(),
                    $bookedSlotDate
                );

                $item->setHasError(true);
                $item->setMessage([$errorMessage]);
                if ($item->getId()) {
                    $item->delete();
                }
            } else {
                $alreadyAddedQty = 0;
                if ($item->getId()) {
                    $collection = $this->quoteCollection->create();
                    $bookingQuote = $helper->getDataByField($item->getId(), 'item_id', $collection);
                    if ($bookingQuote) {
                        $alreadyAddedQty = $item->getQty();
                    }
                }

                if ($requestedQty > $availableQty || $alreadyAddedQty > $availableQty) {
                    $errorMessage = __(
                        'Only %1 quantity is available for %2 for slot %3.',
                        $availableQty,
                        $item->getName(),
                        $bookedSlotDate
                    );

                    $this->checkoutSession->getQuote()->setHasError(true);
                    throw new \Magento\Framework\Exception\LocalizedException(
                        $errorMessage
                    );
                }
            }
        }
    }
}
