<?php
/**
 * Webkul Software.
 *
 * @category  Webkul
 * @package   Webkul_AdvancedBookingSystem
 * @author    Webkul Software Private Limited
 * @copyright Webkul Software Private Limited (https://webkul.com)
 * @license   https://store.webkul.com/license.html
 */
namespace Webkul\AdvancedBookingSystem\Observer;

use Magento\Framework\Event\ObserverInterface;
use Magento\Checkout\Model\Session as CheckoutSession;
use Magento\Framework\App\RequestInterface;
use Magento\Framework\Message\ManagerInterface;
use Webkul\AdvancedBookingSystem\Model\ResourceModel\Quote\CollectionFactory as QuoteCollection;

class CheckoutCartSaveBeforeObserver implements ObserverInterface
{
    /**
     * @var CheckoutSession
     */
    private $checkoutSession;

    /**
     * @var RequestInterface
     */
    private $request;

    /**
     * @var \Webkul\AdvancedBookingSystem\Helper\Data
     */
    private $helper;

    /**
     * @var QuoteCollection
     */
    private $quoteCollection;

    /**
     * Constructor
     *
     * @param CheckoutSession                           $checkoutSession
     * @param RequestInterface                          $request
     * @param \Webkul\AdvancedBookingSystem\Helper\Data $helper
     * @param QuoteCollection                           $quoteCollectionFactory
     */
    public function __construct(
        CheckoutSession $checkoutSession,
        RequestInterface $request,
        \Webkul\AdvancedBookingSystem\Helper\Data $helper,
        QuoteCollection $quoteCollectionFactory
    ) {
        $this->checkoutSession = $checkoutSession;
        $this->request = $request;
        $this->helper = $helper;
        $this->quoteCollection = $quoteCollectionFactory;
    }

    /**
     * Checkout cart product add event handler.
     *
     * @param \Magento\Framework\Event\Observer $observer
     */
    public function execute(\Magento\Framework\Event\Observer $observer)
    {
        $this->helper->logDataInLogger('CheckoutCartSaveBeforeObserver ');
        $data = $this->request->getParams();
        $helper = $this->helper;
        $quoteId =  $this->checkoutSession->getQuote()->getId();
        $items =  $this->checkoutSession->getQuote()->getAllVisibleItems();
        $allowedAttrSetIDs = $helper->getAllowedAttrSetIDs();
        $appointmentAttrSetId = $helper->getProductAttributeSetIdByLabel(
            'Appointment Booking'
        );
        $eventAttrSetId = $helper->getProductAttributeSetIdByLabel(
            'Event Booking'
        );
        $rentalAttrSetId = $helper->getProductAttributeSetIdByLabel(
            'Rental Booking'
        );
        $hotelAttrSetId = $helper->getProductAttributeSetIdByLabel(
            'Hotel Booking'
        );
        $tableAttrSetId = $helper->getProductAttributeSetIdByLabel(
            'Table Booking'
        );
        foreach ($items as $item) {
            $productId = $item->getProductId();
            $product = $helper->getProduct($productId);
            $itemId = (int) $item->getId();
            $collection = $this->quoteCollection->create();
            $bookingQuote = $helper->getDataByField($itemId, 'item_id', $collection);
            $productSetId = $product->getAttributeSetId();
            if ($helper->isBookingProduct($productId) && $itemId && !$bookingQuote) {
                $productSetId = $product->getAttributeSetId();
                if (in_array($productSetId, $allowedAttrSetIDs) || $productSetId==$hotelAttrSetId) {
                    $isThrowError = 1;
                    if ($appointmentAttrSetId == $productSetId) {
                        $helper->processBookingSave($data, $product, $item, $isThrowError);
                    } elseif ($eventAttrSetId == $productSetId) {
                        $helper->processEventBookingSave($data, $product, $item, $isThrowError);
                    } elseif ($rentalAttrSetId == $productSetId) {
                        $helper->processRentBookingSave($data, $product, $item, $isThrowError);
                    } elseif ($productSetId==$hotelAttrSetId) {
                        $helper->processHotelBookingSave(
                            $data,
                            $product,
                            $item,
                            $isThrowError
                        );
                    } elseif ($productSetId==$tableAttrSetId) {
                        $helper->processTableBookingSave($data, $product, $item, $isThrowError);
                    }
                } elseif (!empty($data['slot_id'])) {
                    $helper->processDefaultSlotData($data, $item, $productId, $itemId);
                }
            } elseif ($bookingQuote && $helper->isBookingProduct($productId)) {
                $actionName = $helper->getFullActionName();
                if ($actionName == "checkout_sidebar_updateItemQty" && $productSetId==$tableAttrSetId) {
                    $optionsData['booking_date'] = $bookingQuote->getSlotDate();
                    $optionsData['booking_time'] = $bookingQuote->getSlotTime();
                    $optionsData['slot_day_index'] = $bookingQuote->getSlotDayIndex();
                    $optionsData['slot_id'] = $bookingQuote->getSlotId();
                    $optionsData['parent_slot_id'] = $bookingQuote->getParentSlotId();
                    $this->checkRequestedQtyAvailableForTable($data, $product, $item, $optionsData);
                }
                
                $bookingQuote->setQty($item->getQty())->save();
                $productSetId = $product->getAttributeSetId();
                if (!in_array($productSetId, $allowedAttrSetIDs) && $productSetId!==$hotelAttrSetId && $itemId) {
                    $helper->processItem($item);
                    $bookingQuote->setQty($item->getQty())->save();
                }
            } elseif (!$itemId && !$bookingQuote && $productSetId==$tableAttrSetId) {
                $this->checkRequestedQtyAvailableForTable($data, $product, $item, $data);
            }
        }
    }

    /**
     * Check Requested Quantity for table
     *
     * @param array $data
     * @param object $product
     * @param object $item
     * @param array $optionsData
     */
    public function checkRequestedQtyAvailableForTable($data, $product, $item, $optionsData)
    {
        if (!empty($optionsData['booking_date']) && !empty($optionsData['booking_time'])) {
            $helper = $this->helper;
            $selectedBookingDate = $optionsData['booking_date'];
            $selectedBookingTime = trim($optionsData['booking_time']);
            $bookedSlotDate = date(
                "d M, Y",
                strtotime($selectedBookingDate)
            )." ".trim($selectedBookingTime);
            $availableQty = 0;
            $productId = $product->getId();
            if (empty($optionsData['slot_day_index'])) {
                $optionsData['slot_day_index'] = 0;
            }
            if (empty($optionsData['slot_id'])) {
                $optionsData['slot_id'] = 0;
            }
            if (empty($optionsData['parent_slot_id'])) {
                $optionsData['parent_slot_id'] = 0;
            }
            $parentSlotId = $optionsData['parent_slot_id'];
            $slotId = $optionsData['slot_id'];
            $slotDayIndex = $optionsData['slot_day_index'];

            $bookedData = $helper->getBookedAppointmentDates($productId);
            
            $bookingInfo = $helper->getBookingInfo($productId);
            $bookingSlotData = $helper->getJsonDecodedString(
                $bookingInfo['info']
            );
            if (!empty($bookingSlotData[$slotDayIndex][$parentSlotId]['slots_info'][$slotId])) {
                $slotData = $bookingSlotData[$slotDayIndex][$parentSlotId]['slots_info'][$slotId];
                if (!empty($slotData['qty'])) {
                    $availableQty = $slotData['qty'];
                }
            }
            if (!empty($bookedData[strtotime($selectedBookingDate)][$selectedBookingTime])) {
                $bookedQty = $bookedData[strtotime($selectedBookingDate)][$selectedBookingTime];
                if ($bookedQty > $availableQty) {
                    $availableQty = 0;
                } else {
                    $availableQty = $availableQty - $bookedQty;
                }
            }
            $requestedQty = $item->getQty();
            if (!$availableQty) {
                $errorMessage = __(
                    '%1 quantity is not available for slot %2.',
                    $item->getName(),
                    $bookedSlotDate
                );

                $item->setHasError(true);
                $item->setMessage([$errorMessage]);
                if ($item->getId()) {
                    $item->delete();
                }
            } else {
                $alreadyAddedQty = 0;
                if ($item->getId()) {
                    $collection = $this->quoteCollection->create();
                    $bookingQuote = $helper->getDataByField($item->getId(), 'item_id', $collection);
                    if ($bookingQuote) {
                        $alreadyAddedQty = $item->getQty();
                    }
                }
                if ($requestedQty > $availableQty || ($alreadyAddedQty > $availableQty)) {
                    $errorMessage = __(
                        'Only %1 quantity is available for %2 for slot %3.',
                        $availableQty,
                        $item->getName(),
                        $bookedSlotDate
                    );

                    $this->checkoutSession->getQuote()->setHasError(true);
                    throw new \Magento\Framework\Exception\LocalizedException(
                        $errorMessage
                    );
                }
            }
        }
    }
}
