<?php

/**
 * Webkul Software.
 *
 * @category  Webkul
 * @package   Webkul_AdvancedBookingSystem
 * @author    Webkul Software Private Limited
 * @copyright Webkul Software Private Limited (https://webkul.com)
 * @license   https://store.webkul.com/license.html
 */

namespace Webkul\AdvancedBookingSystem\Plugin\Model\Sales\AdminOrder;

use Webkul\AdvancedBookingSystem\Model\Info;

/**
 * Quote Product Add
 */
class Create
{
    /**
     * Quote session object
     *
     * @var \Magento\Backend\Model\Session\Quote
     */
    private $quoteSession;

    /**
     * @var \Webkul\AdvancedBookingSystem\Helper\Admin\Data
     */
    private $adminHelper;

    /**
     * @var \Magento\Catalog\Model\ProductFactory
     */
    private $productFactory;
    
    /**
     * @var \Magento\Framework\Message\ManagerInterface
     */
    private $messageManager;

    /**
     * @param \Magento\Backend\Model\Session\Quote $quoteSession
     * @param \Webkul\AdvancedBookingSystem\Helper\Admin\Data $adminHelper
     * @param \Magento\Catalog\Model\ProductFactory $productFactory
     * @param \Magento\Framework\Message\ManagerInterface $messageManager
     */
    public function __construct(
        \Magento\Backend\Model\Session\Quote $quoteSession,
        \Webkul\AdvancedBookingSystem\Helper\Admin\Data $adminHelper,
        \Magento\Catalog\Model\ProductFactory $productFactory,
        \Magento\Framework\Message\ManagerInterface $messageManager
    ) {
        $this->quoteSession = $quoteSession;
        $this->adminHelper = $adminHelper;
        $this->productFactory = $productFactory;
        $this->messageManager = $messageManager;
    }

    /**
     * Add quote item custom price.
     *
     * Custom price is updated in case 'custom_price' value exists
     *
     * @param \Magento\Sales\Model\AdminOrder\Create $subject
     * @param int|\Magento\Catalog\Model\Product $product
     * @param array|float|int|\Magento\Framework\DataObject $config
     * @throws \Magento\Framework\Exception\LocalizedException
     * @return Updater
     */
    public function beforeAddProduct(
        \Magento\Sales\Model\AdminOrder\Create $subject,
        $product,
        $config
    ) {
        try {
            $helper = $this->adminHelper;
            if (!$product instanceof \Magento\Catalog\Model\Product) {
                $productId = $product;
                $product = $this->productFactory->create()
                    ->setStore(
                        $this->quoteSession->getStore()
                    )->setStoreId(
                        $this->quoteSession->getStoreId()
                    )->load(
                        $product
                    );
                if (!$product->getId()) {
                    throw new \Magento\Framework\Exception\LocalizedException(
                        __('We could not add a product to cart by the ID "%1".', $productId)
                    );
                }
            }
            $productType = $product->getTypeId();
            if ($productType == 'booking') {
                $productSetId = $product->getAttributeSetId();
                $allowedAttrSetIDs = $helper->getAllowedAttrSetIDs();
                $rentalAttrSetId = $helper->getProductAttributeSetIdByLabel(
                    'Rental Booking'
                );
                if (in_array($productSetId, $allowedAttrSetIDs)) {
                    if ($productSetId == $rentalAttrSetId) {
                        $config = $this->rentTypeBookingAction($product, $config);
                    }
                }
            }
            if ($productType == "hotelbooking") {
                $config = $this->hotelTypeBookingAction($product, $config);
            }
        } catch (\Exception $e) {
            $this->adminHelper->logDataInLogger(
                "admin-order-model-create-addproduct Exception : " . $e->getMessage()
            );
        }
        return [$product, $config];
    }

    /**
     * Add multiple products to current order quote
     *
     * @param \Magento\Sales\Model\AdminOrder\Create $subject
     * @param callable $proceed
     * @param array $products
     * @return $this
     */
    public function aroundAddProducts(\Magento\Sales\Model\AdminOrder\Create $subject, callable $proceed, $products)
    {
        foreach ($products as $productId => $config) {
            $config['qty'] = isset($config['qty']) ? (float)$config['qty'] : 1;
            if (!$productId instanceof \Magento\Catalog\Model\Product) {
                $product = $this->productFactory->create()
                    ->setStore(
                        $this->quoteSession->getStore()
                    )->setStoreId(
                        $this->quoteSession->getStoreId()
                    )->load(
                        $productId
                    );
                if (!$product->getId()) {
                    throw new \Magento\Framework\Exception\LocalizedException(
                        __('We could not add a product to cart by the ID "%1".', $productId)
                    );
                }
            }
            $productType = $product->getTypeId();
            $productSetId = $product->getAttributeSetId();
            $eventAttrSetId = $this->adminHelper->getProductAttributeSetIdByLabel(
                'Event Booking'
            );
            if ($productType == 'booking' && $productSetId == $eventAttrSetId && isset($config['booking_options'])) {
                $eventOptionId =  array_key_first($config['booking_options']);
                $ticketOptions = $config['booking_options'][$eventOptionId];
                foreach ($ticketOptions as $key => $value) {
                    $configInside = $config;
                    if (!$value) {
                        continue;
                    }
                    $configInside['booking_options'][$eventOptionId] = [$key => $value];
                    $configInside['options'][$eventOptionId] = [(string)$key];
                    $configInside['qty'] = $value;
                    try {
                        $subject->addProduct($productId, $configInside);
                    } catch (\Magento\Framework\Exception\LocalizedException $e) {
                        $this->messageManager->addErrorMessage($e->getMessage());
                    } catch (\Exception $e) {
                        return $e;
                    }
                }
            } else {
                try {
                    $subject->addProduct($product, $config);
                } catch (\Magento\Framework\Exception\LocalizedException $e) {
                    $this->messageManager->addErrorMessage($e->getMessage());
                } catch (\Exception $e) {
                    return $e;
                }
            }
        }

        return $this;
    }

    /**
     * Rent Type Booking Action.
     *
     * @param \Magento\Catalog\Model\Product $product
     * @param array $config
     * @return array
     */
    public function rentTypeBookingAction($product, $config)
    {
        $helper = $this->adminHelper;
        $productId = $product->getId();
        $rentOpt = $helper->getRentOptions($product);
        if (!empty($rentOpt['choose_rent_type']['option_id']) && !empty($config['booking_date_from'])) {
            $rentType = '';
            $optionId = $rentOpt['choose_rent_type']['option_id'];
            $optionValues = $rentOpt['choose_rent_type']['option_values'];
            if (!empty($config['options'][$optionId]) && !empty($optionValues)) {
                $optionValId = $config['options'][$optionId];
                foreach ($optionValues as $key => $value) {
                    if ($optionValId == $value['option_type_id']) {
                        if ($value['title'] == 'Hourly Basis') {
                            $rentType = Info::RENT_TYPE_HOURLY;
                        } else {
                            $rentType = Info::RENT_TYPE_DAILY;
                        }
                        break;
                    }
                }
            }
            $bookedSlotFromDate = $config['booking_date_from'];
            if ($rentType == Info::RENT_TYPE_HOURLY) {
                $bookedSlotToDate = $config['booking_date_from'];
                $selectedBookingFromTime = '';
                $selectedBookingToTime = '';
            } else {
                $bookedSlotToDate = $config['booking_date_to'];
                $selectedBookingFromTime = date(
                    "h:i a",
                    strtotime($bookedSlotFromDate)
                );
                $selectedBookingToTime = date(
                    "h:i a",
                    strtotime($bookedSlotToDate)
                );
            }
            $selectedBookingFromDate = date(
                "Y-m-d",
                strtotime($bookedSlotFromDate)
            );
            $selectedBookingToDate = date(
                "Y-m-d",
                strtotime($bookedSlotToDate)
            );
            if ($rentType == Info::RENT_TYPE_HOURLY) {
                if (empty($config['slot_day_index'])) {
                    $config['parent_slot_id'] = 0;
                    $config['slot_id'] = 0;
                    $config['slot_day_index'] = 0;
                    $config['booking_from_time'] = 0;
                    $config['booking_to_time'] = 0;
                }
                $parentSlotId = $config['parent_slot_id'];
                $slotId = $config['slot_id'];
                $slotDayIndex = $config['slot_day_index'];
                $slotIdFrom = $config['booking_from_time'];
                $slotIdTo = $config['booking_to_time'];

                $bookingInfo = $helper->getBookingInfo($productId);
                $bookingSlotData = $helper->getJsonDecodedString(
                    $bookingInfo['info']
                );

                $isSlotExisted = 0;
                if (!empty($bookingSlotData[$slotDayIndex][$parentSlotId]['slots_info'][$slotIdFrom])
                    && !empty($bookingSlotData[$slotDayIndex][$parentSlotId]['slots_info'][$slotIdTo])
                ) {
                    $isSlotExisted = 1;
                }
                if ($config['slot_day_index'] && $isSlotExisted) {
                    $bookingSlotDataArr = $bookingSlotData[$slotDayIndex][$parentSlotId];
                    $slotDataFrom = $bookingSlotDataArr['slots_info'][$slotIdFrom];
                    $slotDataTo = $bookingSlotDataArr['slots_info'][$slotIdTo];
                    if (!empty($slotDataFrom['time']) && !empty($slotDataTo['time'])) {
                        $selectedBookingFromTime = $slotDataFrom['time'];
                        $selectedBookingToTime = $slotDataTo['time'];
                    }
                    // number of hours for rent
                    $hourDiff = strtotime($selectedBookingToTime) - strtotime($selectedBookingFromTime);
                    $rentPeriod = round($hourDiff / (60 * 60));
                    if (!$rentPeriod) {
                        $rentPeriod = 1;
                    }
                    // update item price
                    $price = $helper->getCovertedPrice($product->getFinalPrice());
                    try {
                        $config['custom_price'] = $price * $rentPeriod;
                        $config['original_custom_price'] = $price * $rentPeriod;
                    } catch (\Exception $e) {
                        $helper->logDataInLogger("setCustomPrice Exception : " . $e->getMessage());
                    }
                }
            } elseif ($rentType == Info::RENT_TYPE_DAILY) {
                // number of days for rent
                $dateDiff = strtotime($config['booking_date_to']) - strtotime($config['booking_date_from']);
                $rentPeriod = round($dateDiff / (60 * 60 * 24));
                if (!$rentPeriod) {
                    $rentPeriod = 1;
                }
                if (strtotime($config['booking_date_to']) != strtotime($config['booking_date_from'])) {
                    $rentPeriod++;
                }
                // update item price
                $price = $helper->getCovertedPrice($product->getFinalPrice());
                try {
                    $config['custom_price'] = $price * $rentPeriod;
                    $config['original_custom_price'] = $price * $rentPeriod;
                } catch (\Exception $e) {
                    $helper->logDataInLogger("setCustomPrice Exception : " . $e->getMessage());
                }
            }
        }

        return $config;
    }

    /**
     * Hotel Type Booking Action.
     *
     * @param \Magento\Catalog\Model\Product $product
     * @param array $config
     * @return array
     */
    public function hotelTypeBookingAction($product, $config)
    {
        $helper = $this->adminHelper;
        if (!empty($config['selected_configurable_option']) && !empty($config['hotel_qty'])) {
            $selectedProduct = $this->productFactory->create()
                ->setStore(
                    $this->quoteSession->getStore()
                )->setStoreId(
                    $this->quoteSession->getStoreId()
                )->load(
                    $config['selected_configurable_option']
                );
            // update item price
            $price = $helper->getCovertedPrice($selectedProduct->getFinalPrice());
            try {
                $config['custom_price'] = $price * $config['hotel_qty'];
                $config['original_custom_price'] = $price * $config['hotel_qty'];
            } catch (\Exception $e) {
                $helper->logDataInLogger("setCustomPrice Exception : " . $e->getMessage());
            }
        }

        return $config;
    }
}
