<?php
/**
 * Webkul Software.
 *
 * @category  Webkul
 * @package   Webkul_AdvancedBookingSystem
 * @author    Webkul Software Private Limited
 * @copyright Webkul Software Private Limited (https://webkul.com)
 * @license   https://store.webkul.com/license.html
 */
namespace Webkul\AdvancedBookingSystem\Setup\Patch\Data;

use Magento\Framework\Setup\Patch\DataPatchInterface;
use Magento\Framework\Setup\Patch\SchemaPatchInterface;
use Magento\Framework\Setup\Patch\PatchRevertableInterface;
use Magento\Framework\Setup\ModuleDataSetupInterface;
use Magento\Eav\Setup\EavSetup;
use Magento\Eav\Setup\EavSetupFactory;
use Magento\Eav\Model\Entity\Attribute\SetFactory;
use Magento\Eav\Model\Config as EavConfig;
use Magento\Catalog\Model\ResourceModel\Eav\AttributeFactory;
use Magento\Eav\Model\Entity\Attribute\ScopedAttributeInterface;
use Magento\Catalog\Model\Product;

/**
 * Patch is mechanism, that allows to do atomic upgrade data changes
 */
class CreateAttributes implements
    DataPatchInterface
{
    /**
     * @var ModuleDataSetupInterface $moduleDataSetup
     */
    private $moduleDataSetup;
    
    /**
     * @var EavSetupFactory $eavSetupFactory
     */
    private $eavSetupFactory;
    
    /**
     * @var AttributeFactory $attributeSetFactory
     */
    private $attributeSetFactory;
    
    /**
     * @var EavConfig $eavConfig
     */
    private $eavConfig;
    
    /**
     * @var AttributeFactory $attributeFactory
     */
    private $attributeFactory;

    /**
     * Constructor
     *
     * @param ModuleDataSetupInterface $moduleDataSetup
     * @param EavSetupFactory $eavSetupFactory
     * @param SetFactory $attributeSetFactory
     * @param EavConfig $eavConfig
     * @param AttributeFactory $attributeFactory
     */
    public function __construct(
        ModuleDataSetupInterface $moduleDataSetup,
        EavSetupFactory $eavSetupFactory,
        SetFactory $attributeSetFactory,
        EavConfig $eavConfig,
        AttributeFactory $attributeFactory
    ) {
        $this->moduleDataSetup = $moduleDataSetup;
        $this->eavSetupFactory = $eavSetupFactory;
        $this->attributeSetFactory = $attributeSetFactory;
        $this->eavConfig = $eavConfig;
        $this->attributeFactory = $attributeFactory;
    }

    /**
     * Do Upgrade
     *
     * @return void
     */
    public function apply()
    {
        
        $this->moduleDataSetup->getConnection()->startSetup();
        
        /* Create Attribute Set */
        $attributeSet = $this->createAttributeSet('Appointment Booking');
        $appointmentAttributeSetId = $attributeSet->getId();
        $appointmentAttributeGroupId = $attributeSet->getDefaultGroupId();

        $attributeSet = $this->createAttributeSet('Rental Booking');
        $rentalAttributeSetId = $attributeSet->getId();
        $rentalAttributeGroupId = $attributeSet->getDefaultGroupId();

        $attributeSet = $this->createAttributeSet('Event Booking');
        $eventAttributeSetId = $attributeSet->getId();
        $eventAttributeGroupId = $attributeSet->getDefaultGroupId();
        
        $attributeSet = $this->createAttributeSet('Hotel Booking');
        $hotelAttributeSetId = $attributeSet->getId();
        $hotelAttributeGroupId = $attributeSet->getDefaultGroupId();
        
        $attributeSet = $this->createAttributeSet('Table Booking');
        $tableAttributeSetId = $attributeSet->getId();
        $tableAttributeGroupId = $attributeSet->getDefaultGroupId();
        
        $entityType = $this->eavConfig->getEntityType(Product::ENTITY);
        $entityTypeId = $entityType->getId();
        
        //creating holiday list attribute
        
        /** @var EavSetup $eavSetup */
        $eavSetup = $this->eavSetupFactory->create(['setup' => $this->moduleDataSetup]);

        $eavSetup->addAttribute(
            Product::ENTITY,
            'wk_holiday_list',
            [
                'type' => 'text',
                'backend' => \Magento\Eav\Model\Entity\Attribute\Backend\ArrayBackend::class,
                'frontend' => '',
                'label' => 'Booking Holidays',
                'input' => 'multiselect',
                'class' => '',
                'source' => \Webkul\AdvancedBookingSystem\Model\Source\Holiday::class,
                'global' => ScopedAttributeInterface::SCOPE_WEBSITE,
                'visible' => true,
                'default' => 0,
                'required' => false,
                'user_defined' => true,
                'searchable' => false,
                'filterable' => false,
                'comparable' => false,
                'visible_on_front' => false,
                'used_in_product_listing' => false,
                'unique' => false,
                'apply_to' => 'booking'
             ]
        );
        $holidayAttrId = $eavSetup->getAttributeId($entityTypeId, 'wk_holiday_list');

        /* Create Attributes for Appointment Type Booking */
        /** @var EavSetup $eavSetup */
        $eavSetup = $this->eavSetupFactory->create(['setup' => $this->moduleDataSetup]);

        $eavSetup->addAttribute(
            Product::ENTITY,
            'phone_number',
            [
                'type' => 'varchar',
                'group' => '',
                'backend' => '',
                'frontend' => '',
                'label' => 'Contact Number',
                'input' => 'text',
                'class' => '',
                'source' => '',
                'global' => ScopedAttributeInterface::SCOPE_GLOBAL,
                'visible' => false,
                'required' => true,
                'user_defined' => false,
                'searchable' => false,
                'filterable' => false,
                'comparable' => false,
                'visible_on_front' => false,
                'used_in_product_listing' => false,
                'unique' => false,
             ]
        );
        $phoneNoAttributeId = $eavSetup->getAttributeId($entityTypeId, 'phone_number');

        $eavSetup->addAttribute(
            Product::ENTITY,
            'location',
            [
                'type' => 'varchar',
                'group' => '',
                'backend' => '',
                'frontend' => '',
                'label' => 'Location',
                'input' => 'text',
                'class' => '',
                'source' => '',
                'global' => ScopedAttributeInterface::SCOPE_GLOBAL,
                'visible' => false,
                'required' => true,
                'user_defined' => true,
                'searchable' => true,
                'filterable' => false,
                'comparable' => true,
                'visible_on_front' => true,
                'used_in_product_listing' => false,
                'unique' => false,
             ]
        );
        $locationAttributeId = $eavSetup->getAttributeId($entityTypeId, 'location');

        $eavSetup->addAttribute(
            Product::ENTITY,
            'show_map_loction',
            [
                'type' => 'int',
                'group' => '',
                'backend' => '',
                'frontend' => '',
                'label' => 'Show Map with Location',
                'input' => 'boolean',
                'class' => '',
                'source' => \Magento\Eav\Model\Entity\Attribute\Source\Boolean::class,
                'default' => 1,
                'global' => ScopedAttributeInterface::SCOPE_GLOBAL,
                'visible' => false,
                'required' => false,
                'user_defined' => true,
                'searchable' => false,
                'filterable' => false,
                'comparable' => false,
                'visible_on_front' => false,
                'used_in_product_listing' => false,
                'unique' => false,
             ]
        );
        $showMapAttributeId = $eavSetup->getAttributeId($entityTypeId, 'show_map_loction');

        $eavSetup->addAttribute(
            Product::ENTITY,
            'show_contact_button_to',
            [
                'type' => 'int',
                'group' => '',
                'backend' => '',
                'frontend' => '',
                'label' => 'Show Contact Button to',
                'input' => 'select',
                'class' => '',
                'source' => \Webkul\AdvancedBookingSystem\Model\Source\ShowContactButtonToOptions::class,
                'default' => 1,
                'global' => ScopedAttributeInterface::SCOPE_GLOBAL,
                'visible' => false,
                'required' => true,
                'user_defined' => true,
                'searchable' => false,
                'filterable' => false,
                'comparable' => false,
                'visible_on_front' => false,
                'used_in_product_listing' => false,
                'unique' => false,
             ]
        );
        $showContactBtnToAttributeId = $eavSetup->getAttributeId(
            $entityTypeId,
            'show_contact_button_to'
        );
        // booking attributes

        $eavSetup->addAttribute(
            Product::ENTITY,
            'slot_duration',
            [
                'type' => 'varchar',
                'group' => '',
                'backend' => '',
                'frontend' => '',
                'label' => 'Slot Duration(Mins)',
                'input' => 'text',
                'class' => '',
                'source' => '',
                'global' => ScopedAttributeInterface::SCOPE_GLOBAL,
                'visible' => false,
                'required' => true,
                'user_defined' => true,
                'searchable' => false,
                'filterable' => false,
                'comparable' => false,
                'visible_on_front' => false,
                'used_in_product_listing' => false,
                'unique' => false,
             ]
        );
        $slotDurationAttrId = $eavSetup->getAttributeId($entityTypeId, 'slot_duration');

        $eavSetup->addAttribute(
            Product::ENTITY,
            'break_time_bw_slot',
            [
                'type' => 'varchar',
                'group' => '',
                'backend' => '',
                'frontend' => '',
                'label' => 'Break Time b/w Slots(Mins)',
                'input' => 'text',
                'class' => '',
                'source' => '',
                'global' => ScopedAttributeInterface::SCOPE_GLOBAL,
                'visible' => false,
                'required' => true,
                'user_defined' => true,
                'searchable' => false,
                'filterable' => false,
                'comparable' => false,
                'visible_on_front' => false,
                'used_in_product_listing' => false,
                'unique' => false,
             ]
        );
        $breakTimeBwSlotAttrId = $eavSetup->getAttributeId(
            $entityTypeId,
            'break_time_bw_slot'
        );

        $eavSetup->addAttribute(
            Product::ENTITY,
            'prevent_scheduling_before',
            [
                'type' => 'varchar',
                'group' => '',
                'backend' => '',
                'frontend' => '',
                'label' => 'Prevent Scheduling(Mins)',
                'input' => 'text',
                'class' => '',
                'source' => '',
                'global' => ScopedAttributeInterface::SCOPE_GLOBAL,
                'visible' => false,
                'required' => true,
                'user_defined' => true,
                'searchable' => false,
                'filterable' => false,
                'comparable' => false,
                'visible_on_front' => false,
                'used_in_product_listing' => false,
                'unique' => false,
             ]
        );
        $preventSchedulingAttrId = $eavSetup->getAttributeId(
            $entityTypeId,
            'prevent_scheduling_before'
        );

        $eavSetup->addAttribute(
            Product::ENTITY,
            'available_every_week',
            [
                'type' => 'int',
                'group' => '',
                'backend' => '',
                'frontend' => '',
                'label' => 'Available Every Week',
                'input' => 'boolean',
                'class' => '',
                'source' => \Magento\Eav\Model\Entity\Attribute\Source\Boolean::class,
                'default' => 1,
                'global' => ScopedAttributeInterface::SCOPE_GLOBAL,
                'visible' => false,
                'required' => false,
                'user_defined' => true,
                'searchable' => false,
                'filterable' => false,
                'comparable' => false,
                'visible_on_front' => false,
                'used_in_product_listing' => false,
                'unique' => false,
             ]
        );
        $availableEveryWeekAttrId = $eavSetup->getAttributeId(
            $entityTypeId,
            'available_every_week'
        );

        $eavSetup->addAttribute(
            Product::ENTITY,
            'booking_available_from',
            [
                'type' => 'varchar',
                'group' => '',
                'backend' => '',
                'frontend' => '',
                'label' => 'Available From',
                'input' => 'date',
                'class' => '',
                'source' => \Magento\Eav\Model\Entity\Attribute\Source\Boolean::class,
                'default' => 1,
                'global' => ScopedAttributeInterface::SCOPE_GLOBAL,
                'visible' => false,
                'required' => false,
                'user_defined' => true,
                'searchable' => false,
                'filterable' => false,
                'comparable' => false,
                'visible_on_front' => false,
                'used_in_product_listing' => false,
                'unique' => false,
             ]
        );
        $availableFromAttrId = $eavSetup->getAttributeId(
            $entityTypeId,
            'booking_available_from'
        );

        $eavSetup->addAttribute(
            Product::ENTITY,
            'booking_available_to',
            [
                'type' => 'varchar',
                'group' => '',
                'backend' => '',
                'frontend' => '',
                'label' => 'Available To',
                'input' => 'date',
                'class' => '',
                'source' => \Magento\Eav\Model\Entity\Attribute\Source\Boolean::class,
                'default' => 1,
                'global' => ScopedAttributeInterface::SCOPE_GLOBAL,
                'visible' => false,
                'required' => false,
                'user_defined' => true,
                'searchable' => false,
                'filterable' => false,
                'comparable' => false,
                'visible_on_front' => false,
                'used_in_product_listing' => false,
                'unique' => false,
             ]
        );
        $availableToAttrId = $eavSetup->getAttributeId(
            $entityTypeId,
            'booking_available_to'
        );

        $eavSetup->addAttribute(
            Product::ENTITY,
            'slot_for_all_days',
            [
                'type' => 'int',
                'group' => '',
                'backend' => '',
                'frontend' => '',
                'label' => 'Same Slot for All Days',
                'input' => 'boolean',
                'class' => '',
                'source' => \Magento\Eav\Model\Entity\Attribute\Source\Boolean::class,
                'default' => 1,
                'global' => ScopedAttributeInterface::SCOPE_GLOBAL,
                'visible' => false,
                'required' => false,
                'user_defined' => true,
                'searchable' => false,
                'filterable' => false,
                'comparable' => false,
                'visible_on_front' => false,
                'used_in_product_listing' => false,
                'unique' => false,
             ]
        );
        $slotForAllDaysAttrId = $eavSetup->getAttributeId(
            $entityTypeId,
            'slot_for_all_days'
        );

        $eavSetup->addAttribute(
            Product::ENTITY,
            'slot_has_quantity',
            [
                'type' => 'int',
                'group' => '',
                'backend' => '',
                'frontend' => '',
                'label' => 'Slot has Quantity',
                'input' => 'boolean',
                'class' => '',
                'source' => \Magento\Eav\Model\Entity\Attribute\Source\Boolean::class,
                'default' => 1,
                'global' => ScopedAttributeInterface::SCOPE_GLOBAL,
                'visible' => false,
                'required' => false,
                'user_defined' => true,
                'searchable' => false,
                'filterable' => false,
                'comparable' => false,
                'visible_on_front' => false,
                'used_in_product_listing' => false,
                'unique' => false,
             ]
        );
        $slotHasQtyAttrId = $eavSetup->getAttributeId(
            $entityTypeId,
            'slot_has_quantity'
        );

        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $appointmentAttributeSetId,
            $appointmentAttributeGroupId,
            $phoneNoAttributeId
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $appointmentAttributeSetId,
            $appointmentAttributeGroupId,
            $locationAttributeId
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $appointmentAttributeSetId,
            $appointmentAttributeGroupId,
            $showMapAttributeId
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $appointmentAttributeSetId,
            $appointmentAttributeGroupId,
            $showContactBtnToAttributeId
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $appointmentAttributeSetId,
            $appointmentAttributeGroupId,
            $slotDurationAttrId
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $appointmentAttributeSetId,
            $appointmentAttributeGroupId,
            $breakTimeBwSlotAttrId
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $appointmentAttributeSetId,
            $appointmentAttributeGroupId,
            $preventSchedulingAttrId
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $appointmentAttributeSetId,
            $appointmentAttributeGroupId,
            $availableEveryWeekAttrId
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $appointmentAttributeSetId,
            $appointmentAttributeGroupId,
            $availableFromAttrId
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $appointmentAttributeSetId,
            $appointmentAttributeGroupId,
            $availableToAttrId
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $appointmentAttributeSetId,
            $appointmentAttributeGroupId,
            $slotForAllDaysAttrId
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $appointmentAttributeSetId,
            $appointmentAttributeGroupId,
            $slotHasQtyAttrId
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $appointmentAttributeSetId,
            $appointmentAttributeGroupId,
            $holidayAttrId
        );

        /* Create Attributes for Rent Type Booking */

        /** @var EavSetup $eavSetup */
        $eavSetup = $this->eavSetupFactory->create(['setup' => $this->moduleDataSetup]);
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $rentalAttributeSetId,
            $rentalAttributeGroupId,
            $phoneNoAttributeId
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $rentalAttributeSetId,
            $rentalAttributeGroupId,
            $locationAttributeId
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $rentalAttributeSetId,
            $rentalAttributeGroupId,
            $showMapAttributeId
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $rentalAttributeSetId,
            $rentalAttributeGroupId,
            $showContactBtnToAttributeId
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $rentalAttributeSetId,
            $rentalAttributeGroupId,
            $availableEveryWeekAttrId
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $rentalAttributeSetId,
            $rentalAttributeGroupId,
            $availableFromAttrId
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $rentalAttributeSetId,
            $rentalAttributeGroupId,
            $availableToAttrId
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $rentalAttributeSetId,
            $rentalAttributeGroupId,
            $preventSchedulingAttrId
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $rentalAttributeSetId,
            $rentalAttributeGroupId,
            $slotForAllDaysAttrId
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $rentalAttributeSetId,
            $rentalAttributeGroupId,
            $slotHasQtyAttrId
        );
        $eavSetup->addAttribute(
            Product::ENTITY,
            'renting_type',
            [
                'type' => 'int',
                'group' => '',
                'backend' => '',
                'frontend' => '',
                'label' => 'Renting Type',
                'input' => 'select',
                'class' => '',
                'source' => \Webkul\AdvancedBookingSystem\Model\Source\RentingTypeOptions::class,
                'default' => 1,
                'global' => ScopedAttributeInterface::SCOPE_WEBSITE,
                'visible' => false,
                'required' => false,
                'user_defined' => true,
                'searchable' => false,
                'filterable' => false,
                'comparable' => false,
                'visible_on_front' => false,
                'used_in_product_listing' => false,
                'unique' => false,
             ]
        );
        $rentingTypeAttrId = $eavSetup->getAttributeId(
            $entityTypeId,
            'renting_type'
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $rentalAttributeSetId,
            $rentalAttributeGroupId,
            $rentingTypeAttrId
        );
        
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $rentalAttributeSetId,
            $rentalAttributeGroupId,
            $holidayAttrId
        );

        /* Create Attributes for Event Type Booking */

        /** @var EavSetup $eavSetup */
        $eavSetup = $this->eavSetupFactory->create(['setup' => $this->moduleDataSetup]);

        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $eventAttributeSetId,
            $eventAttributeGroupId,
            $phoneNoAttributeId
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $eventAttributeSetId,
            $eventAttributeGroupId,
            $locationAttributeId
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $eventAttributeSetId,
            $eventAttributeGroupId,
            $showMapAttributeId
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $eventAttributeSetId,
            $eventAttributeGroupId,
            $showContactBtnToAttributeId
        );
        $eavSetup->addAttribute(
            Product::ENTITY,
            'event_chart_available',
            [
                'type' => 'int',
                'group' => '',
                'backend' => '',
                'frontend' => '',
                'label' => 'Event Map/Chart Available',
                'input' => 'boolean',
                'class' => '',
                'source' => \Magento\Eav\Model\Entity\Attribute\Source\Boolean::class,
                'default' => 1,
                'global' => ScopedAttributeInterface::SCOPE_GLOBAL,
                'visible' => false,
                'required' => false,
                'user_defined' => true,
                'searchable' => false,
                'filterable' => false,
                'comparable' => false,
                'visible_on_front' => false,
                'used_in_product_listing' => false,
                'unique' => false,
             ]
        );
        $eventChartAvailableAttrId = $eavSetup->getAttributeId(
            $entityTypeId,
            'event_chart_available'
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $eventAttributeSetId,
            $eventAttributeGroupId,
            $eventChartAvailableAttrId
        );

        $eavSetup->addAttribute(
            Product::ENTITY,
            'event_chart_image',
            [
                'type' => 'varchar',
                'group' => '',
                'backend' => \Webkul\AdvancedBookingSystem\Model\Source\Product\Attribute\Backend\Image::class,
                'frontend' => '',
                'label' => 'Event Chart Image',
                'input' => 'image',
                'class' => '',
                'source' => '',
                'global' => ScopedAttributeInterface::SCOPE_GLOBAL,
                'visible' => false,
                'required' => false,
                'user_defined' => true,
                'searchable' => false,
                'filterable' => false,
                'comparable' => false,
                'visible_on_front' => false,
                'used_in_product_listing' => false,
                'unique' => false,
             ]
        );
        $eventChartImageAttrId = $eavSetup->getAttributeId(
            $entityTypeId,
            'event_chart_image'
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $eventAttributeSetId,
            $eventAttributeGroupId,
            $eventChartImageAttrId
        );

        $eavSetup->addAttribute(
            Product::ENTITY,
            'price_charged_per',
            [
                'type' => 'int',
                'group' => '',
                'backend' => '',
                'frontend' => '',
                'label' => 'Price Charged Per',
                'input' => 'select',
                'class' => '',
                'source' => \Webkul\AdvancedBookingSystem\Model\Source\PriceChargedPerOptions::class,
                'default' => 1,
                'global' => ScopedAttributeInterface::SCOPE_WEBSITE,
                'visible' => false,
                'required' => false,
                'user_defined' => true,
                'searchable' => false,
                'filterable' => false,
                'comparable' => false,
                'visible_on_front' => false,
                'used_in_product_listing' => false,
                'unique' => false,
             ]
        );
        $priceChargedPerAttrId = $eavSetup->getAttributeId(
            $entityTypeId,
            'price_charged_per'
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $eventAttributeSetId,
            $eventAttributeGroupId,
            $priceChargedPerAttrId
        );

        $eavSetup->addAttribute(
            Product::ENTITY,
            'is_multiple_tickets',
            [
                'type' => 'int',
                'group' => '',
                'backend' => '',
                'frontend' => '',
                'label' => 'Event Map/Chart Available',
                'input' => 'boolean',
                'class' => '',
                'source' => \Magento\Eav\Model\Entity\Attribute\Source\Boolean::class,
                'default' => 1,
                'global' => ScopedAttributeInterface::SCOPE_GLOBAL,
                'visible' => false,
                'required' => false,
                'user_defined' => true,
                'searchable' => false,
                'filterable' => false,
                'comparable' => false,
                'visible_on_front' => false,
                'used_in_product_listing' => false,
                'unique' => false,
             ]
        );
        $isMulipleTicketsAttrId = $eavSetup->getAttributeId(
            $entityTypeId,
            'is_multiple_tickets'
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $eventAttributeSetId,
            $eventAttributeGroupId,
            $isMulipleTicketsAttrId
        );
        $eavSetup->addAttribute(
            Product::ENTITY,
            'event_date_from',
            [
                'type' => 'varchar',
                'group' => '',
                'backend' => \Magento\Eav\Model\Entity\Attribute\Backend\Datetime::class,
                'frontend' => '',
                'label' => 'Event Date From',
                'input' => 'datetime',
                'class' => '',
                'source' => '',
                'default' => 1,
                'global' => ScopedAttributeInterface::SCOPE_GLOBAL,
                'visible' => false,
                'required' => false,
                'user_defined' => true,
                'searchable' => false,
                'filterable' => false,
                'comparable' => false,
                'visible_on_front' => false,
                'used_in_product_listing' => false,
                'unique' => false,
             ]
        );
        $eventDateFromAttrId = $eavSetup->getAttributeId(
            $entityTypeId,
            'event_date_from'
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $eventAttributeSetId,
            $eventAttributeGroupId,
            $eventDateFromAttrId
        );

        $eavSetup->addAttribute(
            Product::ENTITY,
            'event_date_to',
            [
                'type' => 'varchar',
                'group' => '',
                'backend' => \Magento\Eav\Model\Entity\Attribute\Backend\Datetime::class,
                'frontend' => '',
                'label' => 'Event Date To',
                'input' => 'datetime',
                'class' => '',
                'source' => '',
                'default' => 1,
                'global' => ScopedAttributeInterface::SCOPE_GLOBAL,
                'visible' => false,
                'required' => false,
                'user_defined' => true,
                'searchable' => false,
                'filterable' => false,
                'comparable' => false,
                'visible_on_front' => false,
                'used_in_product_listing' => false,
                'unique' => false,
             ]
        );
        $eventDateToAttrId = $eavSetup->getAttributeId(
            $entityTypeId,
            'event_date_to'
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $eventAttributeSetId,
            $eventAttributeGroupId,
            $eventDateToAttrId
        );

        /* Create Attributes for Hotel Type Booking */

        /** @var EavSetup $eavSetup */
        $eavSetup = $this->eavSetupFactory->create(['setup' => $this->moduleDataSetup]);
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $hotelAttributeSetId,
            $hotelAttributeGroupId,
            $phoneNoAttributeId
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $hotelAttributeSetId,
            $hotelAttributeGroupId,
            $locationAttributeId
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $hotelAttributeSetId,
            $hotelAttributeGroupId,
            $showMapAttributeId
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $hotelAttributeSetId,
            $hotelAttributeGroupId,
            $showContactBtnToAttributeId
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $hotelAttributeSetId,
            $hotelAttributeGroupId,
            $priceChargedPerAttrId
        );

        $eavSetup->addAttribute(
            Product::ENTITY,
            'ask_a_ques_enable',
            [
                'type' => 'int',
                'group' => '',
                'backend' => '',
                'frontend' => '',
                'label' => 'Enable Ask a Question',
                'input' => 'boolean',
                'class' => '',
                'source' => \Magento\Eav\Model\Entity\Attribute\Source\Boolean::class,
                'default' => 1,
                'global' => ScopedAttributeInterface::SCOPE_GLOBAL,
                'visible' => false,
                'required' => false,
                'user_defined' => true,
                'searchable' => false,
                'filterable' => false,
                'comparable' => false,
                'visible_on_front' => false,
                'used_in_product_listing' => false,
                'unique' => false,
             ]
        );
        $enableAskQuestionAttrId = $eavSetup->getAttributeId(
            $entityTypeId,
            'ask_a_ques_enable'
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $hotelAttributeSetId,
            $hotelAttributeGroupId,
            $enableAskQuestionAttrId
        );

        $eavSetup->addAttribute(
            Product::ENTITY,
            'show_nearby_map',
            [
                'type' => 'int',
                'group' => '',
                'backend' => '',
                'frontend' => '',
                'label' => 'Show Nearby Map',
                'input' => 'boolean',
                'class' => '',
                'source' => \Magento\Eav\Model\Entity\Attribute\Source\Boolean::class,
                'default' => 1,
                'global' => ScopedAttributeInterface::SCOPE_GLOBAL,
                'visible' => false,
                'required' => false,
                'user_defined' => true,
                'searchable' => false,
                'filterable' => false,
                'comparable' => false,
                'visible_on_front' => false,
                'used_in_product_listing' => false,
                'unique' => false,
             ]
        );
        $showNearbyAttrId = $eavSetup->getAttributeId(
            $entityTypeId,
            'show_nearby_map'
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $hotelAttributeSetId,
            $hotelAttributeGroupId,
            $showNearbyAttrId
        );

        $eavSetup->addAttribute(
            Product::ENTITY,
            'hotel_address',
            [
                'type' => 'varchar',
                'group' => '',
                'backend' => '',
                'frontend' => '',
                'label' => 'Address',
                'input' => 'text',
                'class' => '',
                'source' => '',
                'global' => ScopedAttributeInterface::SCOPE_GLOBAL,
                'visible' => false,
                'required' => true,
                'user_defined' => true,
                'searchable' => true,
                'filterable' => false,
                'comparable' => true,
                'visible_on_front' => true,
                'used_in_product_listing' => false,
                'unique' => false,
             ]
        );
        $hotelAddressAttrId = $eavSetup->getAttributeId(
            $entityTypeId,
            'hotel_address'
        );

        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $hotelAttributeSetId,
            $hotelAttributeGroupId,
            $hotelAddressAttrId
        );

        $eavSetup->addAttribute(
            Product::ENTITY,
            'hotel_country',
            [
                'type' => 'varchar',
                'group' => '',
                'backend' => '',
                'frontend' => '',
                'label' => 'Country',
                'input' => 'select',
                'class' => '',
                'source' => \Webkul\AdvancedBookingSystem\Model\Source\Country::class,
                'global' => ScopedAttributeInterface::SCOPE_WEBSITE,
                'visible' => false,
                'required' => true,
                'user_defined' => true,
                'searchable' => true,
                'filterable' => false,
                'comparable' => true,
                'visible_on_front' => true,
                'used_in_product_listing' => false,
                'unique' => false,
             ]
        );
        $hotelCountryAttributeId = $eavSetup->getAttributeId(
            $entityTypeId,
            'hotel_country'
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $hotelAttributeSetId,
            $hotelAttributeGroupId,
            $hotelCountryAttributeId
        );
        
        $eavSetup->addAttribute(
            Product::ENTITY,
            'hotel_state',
            [
                'type' => 'varchar',
                'group' => '',
                'backend' => '',
                'frontend' => '',
                'label' => 'State',
                'input' => 'text',
                'class' => '',
                'global' => ScopedAttributeInterface::SCOPE_GLOBAL,
                'visible' => false,
                'required' => true,
                'user_defined' => true,
                'searchable' => true,
                'filterable' => false,
                'comparable' => true,
                'visible_on_front' => true,
                'used_in_product_listing' => false,
                'unique' => false,
             ]
        );
        $hotelStateAttributeId = $eavSetup->getAttributeId(
            $entityTypeId,
            'hotel_state'
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $hotelAttributeSetId,
            $hotelAttributeGroupId,
            $hotelStateAttributeId
        );

        $eavSetup->addAttribute(
            Product::ENTITY,
            'price_charged_per_hotel',
            [
                'type' => 'int',
                'group' => '',
                'backend' => '',
                'frontend' => '',
                'label' => 'Price Charged Per',
                'input' => 'select',
                'class' => '',
                'source' => \Webkul\AdvancedBookingSystem\Model\Source\PriceChargedPerOptionsHotel::class,
                'default' => 1,
                'global' => ScopedAttributeInterface::SCOPE_WEBSITE,
                'visible' => false,
                'required' => false,
                'user_defined' => true,
                'searchable' => false,
                'filterable' => false,
                'comparable' => false,
                'visible_on_front' => false,
                'used_in_product_listing' => false,
                'unique' => false,
             ]
        );
        $priceChargedPerAttrId = $eavSetup->getAttributeId(
            $entityTypeId,
            'price_charged_per_hotel'
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $hotelAttributeSetId,
            $hotelAttributeGroupId,
            $priceChargedPerAttrId
        );

        $attributeId = $eavSetup->getAttribute(Product::ENTITY, 'amenities', 'attribute_id');
        if (!$attributeId) {
            $eavSetup->addAttribute(
                Product::ENTITY,
                'amenities',
                [
                    'type' => 'varchar',
                    'group' => '',
                    'frontend' => '',
                    'label' => 'Amenities',
                    'input' => 'multiselect',
                    'class' => '',
                    'backend' => \Magento\Eav\Model\Entity\Attribute\Backend\ArrayBackend::class,
                    'default' => 1,
                    'global' => ScopedAttributeInterface::SCOPE_GLOBAL,
                    'visible' => true,
                    'required' => false,
                    'user_defined' => true,
                    'searchable' => true,
                    'filterable' => true,
                    'comparable' => true,
                    'visible_on_front' => true,
                    'used_in_product_listing' => false,
                    'unique' => false,
                    'option' => [
                        'values' => [
                            'Parking',
                            'Beds',
                            'Garden',
                            'Doorstep Doctor',
                            'Free Cycle',
                            'Shower',
                            'Guards And Security',
                            'Doorstep Electrician',
                            'Fixed Phone Line',
                            'Power Backup',
                            'Pets Allowed'
                        ],
                    ],
                 ]
            );
            $amenitiesAttrId = $eavSetup->getAttributeId(
                $entityTypeId,
                'amenities'
            );
            $eavSetup->addAttributeToSet(
                $entityTypeId,
                $hotelAttributeSetId,
                $hotelAttributeGroupId,
                $amenitiesAttrId
            );
        }

        $attributeId = $eavSetup->getAttribute(Product::ENTITY, 'room_type', 'attribute_id');
        if (!$attributeId) {
            $eavSetup->addAttribute(
                Product::ENTITY,
                'room_type',
                [
                    'type' => 'int',
                    'group' => '',
                    'frontend' => '',
                    'label' => 'Room Type',
                    'input' => 'select',
                    'global' => ScopedAttributeInterface::SCOPE_GLOBAL,
                    'visible' => true,
                    'required' => false,
                    'user_defined' => true,
                    'searchable' => true,
                    'filterable' => true,
                    'comparable' => true,
                    'visible_on_front' => false,
                    'used_in_product_listing' => true,
                    'unique' => false,
                    'option' => [
                        'values' => [
                            'Mini Suite',
                            'Queen Suite',
                            'King Suite',
                        ],
                    ],
                 ]
            );
        }

        $eavSetup->addAttribute(
            Product::ENTITY,
            'check_in_time',
            [
                'type' => 'varchar',
                'group' => '',
                'frontend' => '',
                'label' => 'Check In',
                'input' => 'text',
                'class' => '',
                'source' => '',
                'global' => ScopedAttributeInterface::SCOPE_GLOBAL,
                'visible' => false,
                'required' => false,
                'user_defined' => true,
                'searchable' => false,
                'filterable' => false,
                'comparable' => false,
                'visible_on_front' => false,
                'used_in_product_listing' => false,
                'unique' => false,
             ]
        );
        $checkInTimeAttrId = $eavSetup->getAttributeId(
            $entityTypeId,
            'check_in_time'
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $hotelAttributeSetId,
            $hotelAttributeGroupId,
            $checkInTimeAttrId
        );

        $eavSetup->addAttribute(
            Product::ENTITY,
            'check_out_time',
            [
                'type' => 'varchar',
                'group' => '',
                'frontend' => '',
                'label' => 'Check Out',
                'input' => 'text',
                'class' => '',
                'source' => '',
                'global' => ScopedAttributeInterface::SCOPE_GLOBAL,
                'visible' => false,
                'required' => false,
                'user_defined' => true,
                'searchable' => false,
                'filterable' => false,
                'comparable' => false,
                'visible_on_front' => false,
                'used_in_product_listing' => false,
                'unique' => false,
             ]
        );
        $checkOutTimeAttrId = $eavSetup->getAttributeId(
            $entityTypeId,
            'check_out_time'
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $hotelAttributeSetId,
            $hotelAttributeGroupId,
            $checkOutTimeAttrId
        );
        
        /* Create Attributes for Table Type Booking */
        /**
         * @var EavSetup $eavSetup
        */
        $eavSetup = $this->eavSetupFactory->create(['setup' => $this->moduleDataSetup]);

        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $tableAttributeSetId,
            $tableAttributeGroupId,
            $phoneNoAttributeId
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $tableAttributeSetId,
            $tableAttributeGroupId,
            $locationAttributeId
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $tableAttributeSetId,
            $tableAttributeGroupId,
            $showMapAttributeId
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $tableAttributeSetId,
            $tableAttributeGroupId,
            $showContactBtnToAttributeId
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $tableAttributeSetId,
            $tableAttributeGroupId,
            $slotDurationAttrId
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $tableAttributeSetId,
            $tableAttributeGroupId,
            $breakTimeBwSlotAttrId
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $tableAttributeSetId,
            $tableAttributeGroupId,
            $preventSchedulingAttrId
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $tableAttributeSetId,
            $tableAttributeGroupId,
            $slotForAllDaysAttrId
        );
        $eavSetup->addAttribute(
            Product::ENTITY,
            'price_charged_per_table',
            [
                'type' => 'int',
                'group' => '',
                'backend' => '',
                'frontend' => '',
                'label' => 'Charged Per',
                'input' => 'select',
                'class' => '',
                'source' => \Webkul\AdvancedBookingSystem\Model\Source\PriceChargedPerOptionsTable::class,
                'default' => 1,
                'global' => ScopedAttributeInterface::SCOPE_WEBSITE,
                'visible' => false,
                'required' => false,
                'user_defined' => true,
                'searchable' => false,
                'filterable' => false,
                'comparable' => false,
                'visible_on_front' => false,
                'used_in_product_listing' => false,
                'unique' => false,
             ]
        );
        $priceChargedPerAttrId = $eavSetup->getAttributeId(
            $entityTypeId,
            'price_charged_per_table'
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $tableAttributeSetId,
            $tableAttributeGroupId,
            $priceChargedPerAttrId
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $tableAttributeSetId,
            $tableAttributeGroupId,
            $hotelAddressAttrId
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $tableAttributeSetId,
            $tableAttributeGroupId,
            $hotelCountryAttributeId
        );
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $tableAttributeSetId,
            $tableAttributeGroupId,
            $hotelStateAttributeId
        );

        $eavSetup->addAttribute(
            Product::ENTITY,
            'max_capacity',
            [
                'type' => 'varchar',
                'group' => '',
                'backend' => '',
                'frontend' => '',
                'label' => 'No. Of Guests Capacity',
                'input' => 'text',
                'class' => '',
                'source' => '',
                'global' => ScopedAttributeInterface::SCOPE_GLOBAL,
                'visible' => false,
                'required' => true,
                'user_defined' => true,
                'searchable' => false,
                'filterable' => false,
                'comparable' => false,
                'visible_on_front' => false,
                'used_in_product_listing' => false,
                'unique' => false,
             ]
        );
        $maxCapacityAttrId = $eavSetup->getAttributeId(
            $entityTypeId,
            'max_capacity'
        );

        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $tableAttributeSetId,
            $tableAttributeGroupId,
            $maxCapacityAttrId
        );
        
        $eavSetup->addAttribute(
            Product::ENTITY,
            'no_of_guests',
            [
                'type' => 'varchar',
                'group' => '',
                'backend' => '',
                'frontend' => '',
                'label' => 'No. Of Guests',
                'input' => 'text',
                'class' => '',
                'source' => '',
                'global' => ScopedAttributeInterface::SCOPE_GLOBAL,
                'visible' => false,
                'default' => 1,
                'required' => true,
                'user_defined' => true,
                'searchable' => false,
                'filterable' => false,
                'comparable' => false,
                'visible_on_front' => false,
                'used_in_product_listing' => false,
                'unique' => false,
             ]
        );
        $noOfGuestsAttrId = $eavSetup->getAttributeId(
            $entityTypeId,
            'no_of_guests'
        );

        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $tableAttributeSetId,
            $tableAttributeGroupId,
            $noOfGuestsAttrId
        );
        
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $tableAttributeSetId,
            $tableAttributeGroupId,
            $holidayAttrId
        );

        /* Create Attributes for Default Type Booking */
        /**
         * @var EavSetup $eavSetup
        */
        
        $eavSetup = $this->eavSetupFactory->create(['setup' => $this->moduleDataSetup]);
        $defaultAttributeSetId = $eavSetup->getDefaultAttributeSetId(Product::ENTITY);
        $defaultAttributeGroupId = $eavSetup->getDefaultAttributeGroupId(Product::ENTITY, $defaultAttributeSetId);
        $entityTypeId = $eavSetup->getEntityTypeId(Product::ENTITY);

        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $defaultAttributeSetId,
            $defaultAttributeGroupId,
            $locationAttributeId
        );
       
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $defaultAttributeSetId,
            $defaultAttributeGroupId,
            $showMapAttributeId
        );

        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $defaultAttributeSetId,
            $defaultAttributeGroupId,
            $showContactBtnToAttributeId
        );
        
        $eavSetup->addAttributeToSet(
            $entityTypeId,
            $defaultAttributeSetId,
            $defaultAttributeGroupId,
            $holidayAttrId
        );

        $this->setPriceAttribute();
        $this->moduleDataSetup->getConnection()->endSetup();
    }

    /**
     * @inheritdoc
     */
    public function getAliases()
    {
        return [];
    }

    /**
     * @inheritdoc
     */
    public static function getDependencies()
    {
        return [

        ];
    }

    /**
     * Create attribute set and return it
     *
     * @param mixed $attributeSetName
     * @return \Magento\Eav\Model\Entity\Attribute\Set
     */
    public function createAttributeSet($attributeSetName)
    {
        $entityType = $this->eavConfig->getEntityType(\Magento\Catalog\Model\Product::ENTITY);
        $entityTypeId = $entityType->getId();
        $defaultSetId = $entityType->getDefaultAttributeSetId();
        $attributeSet = $this->attributeSetFactory->create();
        $setCollection = $attributeSet->getResourceCollection()
            ->addFieldToFilter('entity_type_id', $entityTypeId)
            ->addFieldToFilter('attribute_set_name', $attributeSetName)
            ->load();
        $attributeSet = $setCollection->fetchItem();

        if (!$attributeSet) {
            $attributeSet = $this->attributeSetFactory->create();
            $attributeSet->setEntityTypeId($entityTypeId);
            $attributeSet->setAttributeSetName($attributeSetName);
            $attributeSet->save();
            $attributeSet->initFromSkeleton($defaultSetId);
            $attributeSet->save();
        }
        return $attributeSet;
    }

    /**
     * Set price attribute
     */
    private function setPriceAttribute()
    {
        try {
            $fieldList = ['price', 'tax_class_id'];
            $eavSetup = $this->eavSetupFactory->create(['setup' => $this->moduleDataSetup]);
            foreach ($fieldList as $field) {
                $applyTo = explode(
                    ',',
                    $eavSetup->getAttribute(\Magento\Catalog\Model\Product::ENTITY, $field, 'apply_to')
                );
                if (!in_array('booking', $applyTo)) {
                    $applyTo[] = 'booking';
                    $eavSetup->updateAttribute(
                        \Magento\Catalog\Model\Product::ENTITY,
                        $field,
                        'apply_to',
                        implode(',', $applyTo)
                    );
                }
            }
        } catch (\Exception $e) {
            $e->getMessage();
        }
    }
}
